
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde
from matplotlib.gridspec import GridSpec

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Seed for reproducibility
np.random.seed(0)
# Generate synthetic art installation data
# Visitors counts at different times of day
morning_visitors = np.random.poisson(200, 500)  # Morning visitors counts
evening_visitors = np.random.poisson(400, 500)  # Evening visitors counts

# Movement speeds through different sections of the exhibition
section1_speeds = np.random.normal(2, 0.5, 1000)  # Speed through section 1 in m/s
section2_speeds = np.random.normal(3, 0.7, 1000)  # Speed through section 2 in m/s

# Elevation data in an art installation
art_route_elevation = np.linspace(0, 500, 1000)
art_elevation_changes = np.sin(np.linspace(0, 10, 1000)) * 20 + art_route_elevation

ax1labels = ["Morning Visitors", "Evening Visitors"]
titles = [
    "Visitor Counts by Time of Day",
    "Movement Speed Distribution by Exhibition Section",
    "Elevation Changes in Art Installation",
]
xlabels = ["Number of Visitors", "Distance (m)"]
ylabels = ["Frequency", "Speed (m/s)", "Elevation (m)"]
ax2xtickslabels = ["Section 1", "Section 2"]
ax2xticks = [1, 2]
bins = np.linspace(50, 550, 31)


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the figure using GridSpec
fig = plt.figure(figsize=(10, 10))
gs = GridSpec(2, 2, figure=fig)

# Histogram plot across top (1,1 and 1,2)
ax1 = fig.add_subplot(gs[0, :])

ax1.hist(
    morning_visitors,
    bins=bins,
    alpha=0.7,
    label=ax1labels[0],
    color="#FFB3BA",
    edgecolor="black",
)
ax1.hist(
    evening_visitors,
    bins=bins,
    alpha=0.7,
    label=ax1labels[1],
    color="#FFDFBA",
    edgecolor="black",
)
ax1.set_title(titles[0])
ax1.set_xlabel(xlabels[0])
ax1.set_ylabel(ylabels[0])
ax1.legend()

# Violin plot on bottom left (2,1)
ax2 = fig.add_subplot(gs[1, 0])
ax2.violinplot([section1_speeds, section2_speeds], showmeans=False, showmedians=True)
ax2.set_title(titles[1])
ax2.set_ylabel(ylabels[1])
ax2.set_xticks(ax2xticks)
ax2.set_xticklabels(ax2xtickslabels)
ax2.grid(True)

# Fill between plot on bottom right (2,2)
ax3 = fig.add_subplot(gs[1, 1])
ax3.fill_between(art_route_elevation, art_elevation_changes, color="#BAE1FF", alpha=0.2)
ax3.set_title(titles[2])
ax3.set_xlabel(xlabels[1])
ax3.set_ylabel(ylabels[2])

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for a better visual appearance
plt.tight_layout()
plt.savefig("multidiff_94.pdf", bbox_inches="tight")
