
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.gridspec as gridspec

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Data for Picture in Picture bar plot
passenger_known_routes = np.array([20000, 15000, 12000, 10000, 8000, 5000, 4000, 3000, 2500, 2000, 1800])
passenger_unknown_routes = np.array([15000, 10000, 8000, 7000, 5000, 3000, 2500, 2000, 1500, 1200, 1000])
bins = np.linspace(0, 0.5, 11)

# Data for the heatmap
satisfaction_ratings = np.array([
    [95, 85, 90, 75, 65, 70],
    [90, 80, 85, 70, 60, 65],
    [85, 75, 80, 65, 55, 60],
    [88, 78, 83, 68, 58, 63],
    [92, 82, 87, 72, 62, 67],
])
x_labels = ["Bus", "Train", "Taxi", "Bike", "Car", "Airplane"]
y_labels = ["North", "South", "East", "West", "Central"]
bar_labels = ["Known Routes", "Unknown Routes"]
xlabels = ["Route Probability", "Transportation Modes"]
ylabels = ["Number of Passengers", "Regions"]
cbarlabel = "Satisfaction Ratings"
insetaxes = [0.251, 0.5, 0.1, 0.2]
insetxlim = [0.25, 0.5]


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with GridSpec
fig = plt.figure(figsize=(10, 5))
gs = gridspec.GridSpec(1, 2, width_ratios=[1, 1.5])

# Picture in Picture bar plot
ax0 = plt.subplot(gs[0])
ax0.bar(
    bins,
    passenger_known_routes,
    width=0.04,
    color="#FF5733",  # Orange for known routes
    align="center",
    label=bar_labels[0],
    edgecolor="black",
)
ax0.bar(
    bins,
    passenger_unknown_routes,
    width=0.04,
    color="#33C3FF",  # Light blue for unknown routes
    align="center",
    bottom=passenger_known_routes,
    label=bar_labels[1],
    edgecolor="black",
)
ax0.set_xlabel(xlabels[0])
ax0.set_ylabel(ylabels[0])
ax0.legend(loc="upper right")
ax0.grid(True)

# Add inset
ax_inset = fig.add_axes(insetaxes)
ax_inset.bar(
    bins[5:],
    passenger_known_routes[5:],
    width=0.04,
    color="#FF5733",
    align="center",
    edgecolor="black",
)
ax_inset.bar(
    bins[5:],
    passenger_unknown_routes[5:],
    width=0.04,
    color="#33C3FF",
    align="center",
    bottom=passenger_known_routes[5:],
    edgecolor="black",
)
ax_inset.set_xlim(insetxlim)

# Heatmap plot
ax1 = plt.subplot(gs[1])
cmap = plt.cm.YlGnBu
norm = plt.Normalize(vmin=satisfaction_ratings.min(), vmax=satisfaction_ratings.max())
heatmap = ax1.imshow(satisfaction_ratings, cmap=cmap, norm=norm, aspect="auto")

# Add color bar
cbar = plt.colorbar(heatmap, ax=ax1, orientation="vertical", pad=0.1)
cbar.set_label(cbarlabel)

# Set x and y labels
ax1.set_xticks(np.arange(len(x_labels)))
ax1.set_yticks(np.arange(len(y_labels)))
ax1.set_xticklabels(x_labels, rotation=45)
ax1.set_yticklabels(y_labels)
ax1.set_xlabel(xlabels[1])
ax1.set_ylabel(ylabels[1])

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("multidiff_98.pdf", bbox_inches="tight")
