# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
# Data for the plot
languages_1 = ["Spanish", "Italian", "Portuguese"]
out_start_1 = [6.2, 7.1, 6.0]
out_group_bias_1 = [-1.44, -2.22, -3.00]
in_start_1 = [5.0, 6.2, 6.5]
in_group_bias_1 = [+1.38, +1.88, +2.88]
ax1_labels = ["Out-group bias\n(Neutrality)", "In-group bias\n(Proximity)"]

languages_2 = ["Russian", "Arabic", "Hindi"]
out_start_2 = [7.5, 6.3, 6.9]
out_group_bias_2 = [-2.75, -1.50, -2.00]
in_start_2 = [7.0, 6.7, 7.2]
in_group_bias_2 = [0.78, 1.25, 1.11]
ax2_labels = ["Out-group bias\n(Proximity)", "In-group bias\n(Neutrality)"]

# Plot titles and labels
xlabel = "Bias Scale"
ylabel = "Languages"
supertitle = "Communication Biases in Different Languages"
title_1 = "European Languages"
title_2 = "Asian Languages"
legend_labels = ["Interaction Type"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(12, 5))
fig.suptitle(supertitle, fontsize=16)

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# Colors for the arrows
out_group_color = "#FF6347"  # Tomato
in_group_color = "#4682B4"  # SteelBlue

# First subplot (languages_1)
for i, lang in enumerate(languages_1):
    # Out-group bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(out_start_1[i], i + offset * 3 / 2),
        xytext=(out_start_1[i] + out_group_bias_1[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color=out_group_color),
    )
    ax1.scatter(
        [out_start_1[i], out_start_1[i] + out_group_bias_1[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{out_group_bias_1[i]:.2f}",
        (out_start_1[i] + out_group_bias_1[i], i + offset * 1.75),
        color=out_group_color,
        ha="right",
        va="center",
    )

    # In-group bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(in_start_1[i], i + offset / 2),
        xytext=(in_start_1[i] + in_group_bias_1[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<-", color=in_group_color),
    )
    ax1.scatter(
        [in_start_1[i], in_start_1[i] + in_group_bias_1[i]],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{in_group_bias_1[i]:.2f}",
        (in_start_1[i] + in_group_bias_1[i], i + offset * 0.75),
        color=in_group_color,
        ha="left",
        va="center",
    )

# Second subplot (languages_2)
for i, lang in enumerate(languages_2):
    ax2.annotate(
        "",
        xy=(out_start_2[i], i + offset * 3 / 2),
        xytext=(out_start_2[i] + out_group_bias_2[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color=out_group_color),
    )
    ax2.scatter(
        [out_start_2[i], out_start_2[i] + out_group_bias_2[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{out_group_bias_2[i]:.2f}",
        (out_start_2[i] + out_group_bias_2[i], i + offset * 1.75),
        color=out_group_color,
        ha="right",
        va="center",
    )

    ax2.annotate(
        "",
        xy=(in_start_2[i], i + offset / 2),
        xytext=(in_start_2[i] + in_group_bias_2[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<-", color=in_group_color),
    )
    ax2.scatter(
        [in_start_2[i], in_start_2[i] + in_group_bias_2[i]],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{in_group_bias_2[i]:.2f}",
        (in_start_2[i] + in_group_bias_2[i], i + offset * 0.75),
        color=in_group_color,
        ha="left",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(languages_1))
ax2.set_ylim(0, len(languages_2))

# Set x-axis limits uniformly
ax1.set_xlim(0, 10)
ax2.set_xlim(0, 10)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(languages_1))])
ax1.set_yticklabels(languages_1)
ax2.set_yticks([i + offset for i in range(len(languages_2))])
ax2.set_yticklabels(languages_2)
ax2.yaxis.tick_right()
ax2.yaxis.set_label_position("right")

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(languages_1))], minor=True)
ax2.set_yticks([i for i in range(len(languages_2))], minor=True)
ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")
ax2.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")

# Add x-axis grid lines and set gap is 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(1))
ax2.xaxis.set_major_locator(plt.MultipleLocator(1))
ax1.grid(axis="x", linestyle="--", linewidth=0.5)
ax2.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
red_arrow_1 = mlines.Line2D(
    [],
    [],
    color=out_group_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax1_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
blue_arrow_1 = mlines.Line2D(
    [],
    [],
    color=in_group_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax1_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(
    handles=[red_arrow_1, blue_arrow_1],
    bbox_to_anchor=(0.45, 0),
    ncol=2,
    title=legend_labels[0],
)

red_arrow_2 = mlines.Line2D(
    [],
    [],
    color=out_group_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax2_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
blue_arrow_2 = mlines.Line2D(
    [],
    [],
    color=in_group_color,
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax2_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[red_arrow_2, blue_arrow_2], bbox_to_anchor=(0.85, 0), ncol=2)

# Set individual titles for the subplots
ax1.set_title(title_1)
ax2.set_title(title_2)

# Add x and y axis labels
fig.text(0.5, 0.04, xlabel, ha="center", va="center")
fig.text(0.06, 0.5, ylabel, ha="center", va="center", rotation="vertical")

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout(rect=[0, 0.03, 1, 0.95])
plt.savefig("quiver_23.pdf", bbox_inches="tight")
