# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines
import matplotlib.patches as mpatches

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)


# Define the vector field function (example: wind flow)
def vector_field(X, Y):
    U = -Y
    V = X
    return U, V


def modified_vector_field(X, Y):
    U = -1 - X**2 + Y
    V = 1 + X - Y**2
    return U, V


# Create a grid of points
x = np.linspace(0, 0.6, 10)
y = np.linspace(0, 0.6, 10)
X, Y = np.meshgrid(x, y)

# Compute the vector field
U, V = vector_field(X, Y)

# Compute the modified vector field
U_mod, V_mod = modified_vector_field(X, Y)

# Plot the curves as inverse functions with slightly different denominators for variation
x = np.linspace(0.2, 0.5, 100)
xlabel = "Time (hours)"
ylabel = "Energy Output (kWh)"
patch_labels = ["True Wind Flow", "Modified Wind Flow"]
line_labels = [
    "Solar Output (Train)",
    "Solar Output (Test)",
    "Wind Output (Train)",
    "Wind Output (Test)",
]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(8, 6))
plt.quiver(X, Y, U, V, color="#4caf50", alpha=0.6)  # Green
plt.quiver(X, Y, U_mod, V_mod, color="#3f51b5", alpha=0.6)  # Blue
plt.plot(
    x, 0.09 / (x**1.2), color="#ff9800", linestyle="-", linewidth=1.5
)  # Orange, solid
plt.plot(
    x, 0.08 / (x**1.2 + 0.04), color="#9c27b0", linestyle="--", linewidth=1.5
)  # Purple, dashed
plt.plot(
    x, 0.075 / (x**1 + 0.04), color="#e91e63", linestyle="-.", linewidth=1.5
)  # Pink, dash-dot
plt.plot(
    x, 0.12 / (x**1 + 0.05), color="#607d8b", linestyle=":", linewidth=1.5
)  # Blue-grey, dotted

# Add labels and legend
plt.xlabel(xlabel, fontsize=14, style="italic")
plt.ylabel(ylabel, fontsize=14, style="italic")

green_patch = mpatches.Patch(color="#4caf50", label=patch_labels[0], alpha=0.6)
blue_patch = mpatches.Patch(color="#3f51b5", label=patch_labels[1], alpha=0.6)

# Create legend for curves
solar_train_line = mlines.Line2D(
    [], [], color="#ff9800", linestyle="-", linewidth=1.5, label=line_labels[0]
)
solar_test_line = mlines.Line2D(
    [], [], color="#9c27b0", linestyle="--", linewidth=1.5, label=line_labels[1]
)
wind_train_line = mlines.Line2D(
    [], [], color="#e91e63", linestyle="-.", linewidth=1.5, label=line_labels[2]
)
wind_test_line = mlines.Line2D(
    [], [], color="#607d8b", linestyle=":", linewidth=1.5, label=line_labels[3]
)

# Combine all legend handles
handles = [
    green_patch,
    blue_patch,
    solar_train_line,
    solar_test_line,
    wind_train_line,
    wind_test_line,
]

# Add the legend to the plot with specified location
plt.legend(handles=handles, loc="lower left")

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("quiver_27.pdf", bbox_inches="tight")
