# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
# Data for the plot
subjects_1 = ["Ethics", "Logic", "Metaphysics"]
out_start_1 = [7.2, 8.0, 6.5]
out_group_bias_1 = [-2.1, -2.7, -1.9]
in_start_1 = [6.8, 7.4, 6.0]
in_group_bias_1 = [+1.8, +1.6, +2.0]
xlabel = "Scores"
ylabel_1 = "Philosophy Domains (Out-group Bias)"
ylabel_2 = "Philosophy Domains (In-group Bias)"
ax1_labels = ["Out-group bias\n(Logics)", "In-group bias\n(Logics)"]

subjects_2 = ["Epistemology", "Aesthetics", "Political Philosophy"]
out_start_2 = [7.7, 6.9, 8.1]
out_group_bias_2 = [-2.4, -1.8, -3.0]
in_start_2 = [7.1, 6.3, 7.5]
in_group_bias_2 = [+1.4, +2.1, +1.8]
ax2_labels = ["Out-group bias\n(Aesthetics)", "In-group bias\n(Aesthetics)"]

title = "Bias Trends in Philosophy Domains"
supertitle = "Comparison of Out-group and In-group Biases in Various Philosophy Domains"
legend_labels_1 = ["Out-group Bias (Logics)", "In-group Bias (Logics)"]
legend_labels_2 = ["Out-group Bias (Aesthetics)", "In-group Bias (Aesthetics)"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(14, 6))

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# First subplot (subjects_1)
for i, subject in enumerate(subjects_1):
    # Out-group bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(out_start_1[i], i + offset * 3 / 2),
        xytext=(out_start_1[i] + out_group_bias_1[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color="seagreen"),
    )
    ax1.scatter(
        [out_start_1[i], out_start_1[i] + out_group_bias_1[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=15,
    )
    ax1.annotate(
        f"{out_group_bias_1[i]:.2f}",
        (out_start_1[i] + out_group_bias_1[i], i + offset * 1.75),
        color="seagreen",
        ha="right",
        va="center",
    )

    # In-group bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(in_start_1[i], i + offset / 2),
        xytext=(in_start_1[i] + in_group_bias_1[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<-", color="darkolivegreen"),
    )
    ax1.scatter(
        [in_start_1[i], in_start_1[i] + in_group_bias_1[i]],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=15,
    )
    ax1.annotate(
        f"{in_group_bias_1[i]:.2f}",
        (in_start_1[i] + in_group_bias_1[i], i + offset * 0.75),
        color="darkolivegreen",
        ha="left",
        va="center",
    )

# Second subplot (subjects_2)
for i, subject in enumerate(subjects_2):
    ax2.annotate(
        "",
        xy=(out_start_2[i], i + offset * 3 / 2),
        xytext=(out_start_2[i] + out_group_bias_2[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color="royalblue"),
    )
    ax2.scatter(
        [out_start_2[i], out_start_2[i] + out_group_bias_2[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=15,
    )
    ax2.annotate(
        f"{out_group_bias_2[i]:.2f}",
        (out_start_2[i] + out_group_bias_2[i], i + offset * 1.75),
        color="royalblue",
        ha="right",
        va="center",
    )

    ax2.annotate(
        "",
        xy=(in_start_2[i], i + offset / 2),
        xytext=(in_start_2[i] + in_group_bias_2[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<-", color="steelblue"),
    )
    ax2.scatter(
        [in_start_2[i], in_start_2[i] + in_group_bias_2[i]],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=15,
    )
    ax2.annotate(
        f"{in_group_bias_2[i]:.2f}",
        (in_start_2[i] + in_group_bias_2[i], i + offset * 0.75),
        color="steelblue",
        ha="left",
        va="center",
    )

# Set titles and labels
ax1.set_ylabel(ylabel_1)
ax2.set_ylabel(ylabel_2)
ax1.set_xlabel(xlabel)
ax2.set_xlabel(xlabel)
fig.suptitle(supertitle, fontsize=14)

# Set y-axis limits
ax1.set_ylim(0, len(subjects_1))
ax2.set_ylim(0, len(subjects_2))

# Set x-axis limits uniformly
ax1.set_xlim(2, 12)
ax2.set_xlim(2, 12)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(subjects_1))])
ax1.set_yticklabels(subjects_1)
ax2.set_yticks([i + offset for i in range(len(subjects_2))])
ax2.set_yticklabels(subjects_2)
ax2.yaxis.tick_right()
ax2.yaxis.set_label_position("right")

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(subjects_1))], minor=True)
ax2.set_yticks([i for i in range(len(subjects_2))], minor=True)
ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")
ax2.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")

# Add x-axis grid lines and set gap is 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(1))
ax2.xaxis.set_major_locator(plt.MultipleLocator(1))
ax1.grid(axis="x", linestyle="--", linewidth=0.5)
ax2.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
seagreen_arrow = mlines.Line2D(
    [],
    [],
    color="seagreen",
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax1_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
darkolivegreen_arrow = mlines.Line2D(
    [],
    [],
    color="darkolivegreen",
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax1_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(
    handles=[seagreen_arrow, darkolivegreen_arrow], bbox_to_anchor=(0.45, 0), ncol=2
)

royalblue_arrow = mlines.Line2D(
    [],
    [],
    color="royalblue",
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax2_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
steelblue_arrow = mlines.Line2D(
    [],
    [],
    color="steelblue",
    marker=">",
    linestyle="-",
    markersize=8,
    label=ax2_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[royalblue_arrow, steelblue_arrow], bbox_to_anchor=(0.85, 0), ncol=2)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("quiver_37.pdf", bbox_inches="tight")
