# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
# Data for the plot (Non-Profit Organizations)
orgs_1 = ["Red Cross", "Doctors Without Borders", "UNICEF"]
public_support_initial_1 = [50, 40, 45]
public_support_change_1 = [10, 8, 9]
funds_initial_1 = [100, 120, 110]
funds_change_1 = [15, 20, 18]
ax1_labels = ["Fund Growth", "Public Support Growth"]

orgs_2 = ["Greenpeace", "Amnesty International", "Save the Children"]
public_support_initial_2 = [35, 30, 25]
public_support_change_2 = [7, 6, 5]
funds_initial_2 = [80, 70, 65]
funds_change_2 = [12, 10, 9]
ax2_labels = ["Fund Growth", "Public Support Growth"]

# Custom Label Texts
xlabel_1 = "Support and Funds (in $ Millions)"
xlabel_2 = "Support and Funds (in $ Millions)"
ylabel_1 = "Organizations"
ylabel_2 = "Organizations"
title_1 = "Growth of Public Support and Funds in 2022\n(Group 1)"
title_2 = "Growth of Public Support and Funds in 2022\n(Group 2)"
suptitle = "Visualization of Non-Profit Organizations' Growth"
legend_labels = ["Fund Growth", "Public Support Growth"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(12, 5))

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# First subplot (orgs_1)
for i, org in enumerate(orgs_1):
    # Fund growth line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(funds_initial_1[i], i + offset * 3 / 2),
        xytext=(funds_initial_1[i] + funds_change_1[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color="mediumseagreen"),
    )
    ax1.scatter(
        [funds_initial_1[i], funds_initial_1[i] + funds_change_1[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{funds_change_1[i]:.2f}",
        (funds_initial_1[i] + funds_change_1[i], i + offset * 1.75),
        color="mediumseagreen",
        ha="right",
        va="center",
    )

    # Public support growth line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(public_support_initial_1[i], i + offset / 2),
        xytext=(
            public_support_initial_1[i] + public_support_change_1[i],
            i + offset / 2,
        ),
        arrowprops=dict(arrowstyle="<-", color="royalblue"),
    )
    ax1.scatter(
        [
            public_support_initial_1[i],
            public_support_initial_1[i] + public_support_change_1[i],
        ],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{public_support_change_1[i]:.2f}",
        (public_support_initial_1[i] + public_support_change_1[i], i + offset * 0.75),
        color="royalblue",
        ha="left",
        va="center",
    )

# Second subplot (orgs_2)
for i, org in enumerate(orgs_2):
    ax2.annotate(
        "",
        xy=(funds_initial_2[i], i + offset * 3 / 2),
        xytext=(funds_initial_2[i] + funds_change_2[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color="mediumseagreen"),
    )
    ax2.scatter(
        [funds_initial_2[i], funds_initial_2[i] + funds_change_2[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{funds_change_2[i]:.2f}",
        (funds_initial_2[i] + funds_change_2[i], i + offset * 1.75),
        color="mediumseagreen",
        ha="right",
        va="center",
    )

    ax2.annotate(
        "",
        xy=(public_support_initial_2[i], i + offset / 2),
        xytext=(
            public_support_initial_2[i] + public_support_change_2[i],
            i + offset / 2,
        ),
        arrowprops=dict(arrowstyle="<-", color="royalblue"),
    )
    ax2.scatter(
        [
            public_support_initial_2[i],
            public_support_initial_2[i] + public_support_change_2[i],
        ],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{public_support_change_2[i]:.2f}",
        (public_support_initial_2[i] + public_support_change_2[i], i + offset * 0.75),
        color="royalblue",
        ha="left",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(orgs_1))
ax2.set_ylim(0, len(orgs_2))

# Set x-axis limits uniformly
ax1.set_xlim(0, 150)
ax2.set_xlim(0, 150)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(orgs_1))])
ax1.set_yticklabels(orgs_1)
ax2.set_yticks([i + offset for i in range(len(orgs_2))])
ax2.set_yticklabels(orgs_2)
ax2.yaxis.tick_right()
ax2.yaxis.set_label_position("right")

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(orgs_1))], minor=True)
ax2.set_yticks([i for i in range(len(orgs_2))], minor=True)
ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")
ax2.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")

# Add x-axis grid lines and set gap to 10
ax1.xaxis.set_major_locator(plt.MultipleLocator(10))
ax2.xaxis.set_major_locator(plt.MultipleLocator(10))
ax1.grid(axis="x", linestyle="--", linewidth=0.5)
ax2.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
green_arrow = mlines.Line2D(
    [],
    [],
    color="mediumseagreen",
    marker=">",
    linestyle="-",
    markersize=8,
    label=legend_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
blue_arrow = mlines.Line2D(
    [],
    [],
    color="royalblue",
    marker=">",
    linestyle="-",
    markersize=8,
    label=legend_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[green_arrow, blue_arrow], bbox_to_anchor=(0.5, 0), ncol=2)

# Set titles and labels
ax1.set_title(title_1)
ax1.set_xlabel(xlabel_1)
ax1.set_ylabel(ylabel_1)

ax2.set_title(title_2)
ax2.set_xlabel(xlabel_2)
ax2.set_ylabel(ylabel_2)

fig.suptitle(suptitle, fontsize=14)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout(rect=[0, 0.03, 1, 0.95])
plt.savefig("quiver_40.pdf", bbox_inches="tight")
