# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
# Data for the plot
# Updated for a new domain: Environmental Science (Renewable Energy Sources and Technologies)
renewables = ["Solar", "Wind", "Hydropower"]
performance_scores_renew = [8.0, 7.5, 9.0]
performance_bias_renew = [+2.5, +3.0, +1.8]
usability_scores_renew = [7.0, 8.5, 6.8]
usability_bias_renew = [-2.0, -1.5, -2.5]
renew_labels = ["Performance Bias", "Usability Bias"]

technologies = ["Photovoltaic", "Turbine", "Dams"]
performance_scores_tech = [7.8, 7.0, 8.5]
performance_bias_tech = [+2.8, +2.5, +2.0]
usability_scores_tech = [6.5, 7.8, 7.0]
usability_bias_tech = [-1.8, -1.2, -2.0]
tech_labels = ["Performance Bias", "Usability Bias"]

# Labels for the chart
xlabel = "Scores"
ylabel_renew = "Renewable Energy Sources"
ylabel_tech = "Technologies"
title_renew = "Renewable Energy: Performance and Usability Biases"
title_tech = "Technologies: Performance and Usability Biases"
supertitle = "Renewable Energy Performance and Usability Biases"
legendlabels_renew = ["Performance Bias", "Usability Bias"]
legendlabels_tech = ["Performance Bias", "Usability Bias"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(14, 6))

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# First subplot (renewables)
for i, tech in enumerate(renewables):
    # Performance bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(performance_scores_renew[i], i + offset * 3 / 2),
        xytext=(
            performance_scores_renew[i] + performance_bias_renew[i],
            i + offset * 3 / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="solid", color="green"),
    )
    ax1.scatter(
        [
            performance_scores_renew[i],
            performance_scores_renew[i] + performance_bias_renew[i],
        ],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{performance_bias_renew[i]:.2f}",
        (performance_scores_renew[i] + performance_bias_renew[i], i + offset * 1.75),
        color="green",
        ha="left",
        va="center",
    )

    # Usability bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(usability_scores_renew[i], i + offset / 2),
        xytext=(usability_scores_renew[i] + usability_bias_renew[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<|-", linestyle="solid", color="blue"),
    )
    ax1.scatter(
        [
            usability_scores_renew[i],
            usability_scores_renew[i] + usability_bias_renew[i],
        ],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{usability_bias_renew[i]:.2f}",
        (usability_scores_renew[i] + usability_bias_renew[i], i + offset * 0.75),
        color="blue",
        ha="right",
        va="center",
    )

# Second subplot (technologies)
for i, tech in enumerate(technologies):
    ax2.annotate(
        "",
        xy=(performance_scores_tech[i], i + offset * 3 / 2),
        xytext=(
            performance_scores_tech[i] + performance_bias_tech[i],
            i + offset * 3 / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="solid", color="green"),
    )
    ax2.scatter(
        [
            performance_scores_tech[i],
            performance_scores_tech[i] + performance_bias_tech[i],
        ],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{performance_bias_tech[i]:.2f}",
        (performance_scores_tech[i] + performance_bias_tech[i], i + offset * 1.75),
        color="green",
        ha="left",
        va="center",
    )

    ax2.annotate(
        "",
        xy=(usability_scores_tech[i], i + offset / 2),
        xytext=(usability_scores_tech[i] + usability_bias_tech[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<|-", linestyle="solid", color="blue"),
    )
    ax2.scatter(
        [usability_scores_tech[i], usability_scores_tech[i] + usability_bias_tech[i]],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{usability_bias_tech[i]:.2f}",
        (usability_scores_tech[i] + usability_bias_tech[i], i + offset * 0.75),
        color="blue",
        ha="right",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(renewables))
ax2.set_ylim(0, len(technologies))

# Set x-axis limits uniformly
ax1.set_xlim(0, 12)
ax2.set_xlim(0, 12)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(renewables))])
ax1.set_yticklabels(renewables)
ax2.set_yticks([i + offset for i in range(len(technologies))])
ax2.set_yticklabels(technologies)
ax2.yaxis.tick_right()
ax2.yaxis.set_label_position("right")

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(renewables))], minor=True)
ax2.set_yticks([i for i in range(len(technologies))], minor=True)
ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="gray")
ax2.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="gray")

# Add x-axis grid lines and set gap to 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(1))
ax2.xaxis.set_major_locator(plt.MultipleLocator(1))
ax1.grid(axis="x", linestyle="--", linewidth=0.5)
ax2.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
green_arrow = mlines.Line2D(
    [],
    [],
    color="green",
    marker=">",
    linestyle="solid",
    markersize=8,
    label=legendlabels_renew[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
blue_arrow = mlines.Line2D(
    [],
    [],
    color="blue",
    marker=">",
    linestyle="solid",
    markersize=8,
    label=legendlabels_renew[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[green_arrow, blue_arrow], bbox_to_anchor=(0.45, 0.0), ncol=2)

green_arrow = mlines.Line2D(
    [],
    [],
    color="green",
    marker=">",
    linestyle="solid",
    markersize=8,
    label=legendlabels_tech[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
blue_arrow = mlines.Line2D(
    [],
    [],
    color="blue",
    marker=">",
    linestyle="solid",
    markersize=8,
    label=legendlabels_tech[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[green_arrow, blue_arrow], bbox_to_anchor=(0.85, 0.0), ncol=2)

# Add titles and labels
ax1.set_xlabel(xlabel)
ax2.set_xlabel(xlabel)
ax1.set_ylabel(ylabel_renew)
ax2.set_ylabel(ylabel_tech)
ax1.set_title(title_renew)
ax2.set_title(title_tech)
plt.suptitle(supertitle)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("quiver_42.pdf", bbox_inches="tight")
