# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
# Data for the plot
# New domain: Transportation Efficiency and User Preferences
vehicles = ["Car", "Bicycle", "Bus"]
fuel_efficiency_scores = [8.5, 10.0, 6.5]
fuel_efficiency_bias = [+2.0, +1.0, +2.0]
user_satisfaction_scores = [7.5, 9.0, 6.0]
user_satisfaction_bias = [-2.0, -2.5, -1.5]
vehicle_labels = ["Fuel Efficiency Bias", "User Satisfaction Bias"]

services = ["Uber", "Public Transit", "Bike Sharing"]
service_efficiency_scores = [8.0, 7.5, 9.0]
service_efficiency_bias = [+2.0, +1.5, +2.5]
service_satisfaction_scores = [7.0, 6.5, 8.0]
service_satisfaction_bias = [-1.5, -2.0, -1.0]
service_labels = ["Service Efficiency Bias", "User Satisfaction Bias"]

# Axis Labels and Other Texts
xlabel = "Score"
ylabel_vehicles = "Vehicles"
ylabel_services = "Services"
title_vehicles = "Vehicle Fuel Efficiency and User Satisfaction"
title_services = "Transportation Service Efficiency and User Satisfaction"
suptitle = "Transportation Performance and Preferences"
legend_labels = vehicle_labels  # Reuse the same labels for both charts

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(12, 6))

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# First subplot (vehicles)
for i, vehicle in enumerate(vehicles):
    # Fuel efficiency bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(fuel_efficiency_scores[i], i + offset * 3 / 2),
        xytext=(
            fuel_efficiency_scores[i] + fuel_efficiency_bias[i],
            i + offset * 3 / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="dotted", color="green"),
    )
    ax1.scatter(
        [
            fuel_efficiency_scores[i],
            fuel_efficiency_scores[i] + fuel_efficiency_bias[i],
        ],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{fuel_efficiency_bias[i]:.2f}",
        (fuel_efficiency_scores[i] + fuel_efficiency_bias[i], i + offset * 1.75),
        color="green",
        ha="left",
        va="center",
    )

    # User satisfaction bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(user_satisfaction_scores[i], i + offset / 2),
        xytext=(
            user_satisfaction_scores[i] + user_satisfaction_bias[i],
            i + offset / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="dashed", color="blue"),
    )
    ax1.scatter(
        [
            user_satisfaction_scores[i],
            user_satisfaction_scores[i] + user_satisfaction_bias[i],
        ],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{user_satisfaction_bias[i]:.2f}",
        (user_satisfaction_scores[i] + user_satisfaction_bias[i], i + offset * 0.75),
        color="blue",
        ha="right",
        va="center",
    )

# Second subplot (services)
for i, service in enumerate(services):
    ax2.annotate(
        "",
        xy=(service_efficiency_scores[i], i + offset * 3 / 2),
        xytext=(
            service_efficiency_scores[i] + service_efficiency_bias[i],
            i + offset * 3 / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="dotted", color="green"),
    )
    ax2.scatter(
        [
            service_efficiency_scores[i],
            service_efficiency_scores[i] + service_efficiency_bias[i],
        ],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{service_efficiency_bias[i]:.2f}",
        (service_efficiency_scores[i] + service_efficiency_bias[i], i + offset * 1.75),
        color="green",
        ha="left",
        va="center",
    )

    ax2.annotate(
        "",
        xy=(service_satisfaction_scores[i], i + offset / 2),
        xytext=(
            service_satisfaction_scores[i] + service_satisfaction_bias[i],
            i + offset / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="dashed", color="blue"),
    )
    ax2.scatter(
        [
            service_satisfaction_scores[i],
            service_satisfaction_scores[i] + service_satisfaction_bias[i],
        ],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{service_satisfaction_bias[i]:.2f}",
        (
            service_satisfaction_scores[i] + service_satisfaction_bias[i],
            i + offset * 0.75,
        ),
        color="blue",
        ha="right",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(vehicles))
ax2.set_ylim(0, len(services))

# Set x-axis limits uniformly
ax1.set_xlim(0, 12)
ax2.set_xlim(0, 12)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(vehicles))])
ax1.set_yticklabels(vehicles)
ax2.set_yticks([i + offset for i in range(len(services))])
ax2.set_yticklabels(services)
ax2.yaxis.tick_right()
ax2.yaxis.set_label_position("right")

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(vehicles))], minor=True)
ax2.set_yticks([i for i in range(len(services))], minor=True)
ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="gray")
ax2.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="gray")

# Add x-axis grid lines and set gap to 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(1))
ax2.xaxis.set_major_locator(plt.MultipleLocator(1))
ax1.grid(axis="x", linestyle="--", linewidth=0.5)
ax2.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
green_arrow = mlines.Line2D(
    [],
    [],
    color="green",
    marker=">",
    linestyle="dotted",
    markersize=8,
    label=legend_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
blue_arrow = mlines.Line2D(
    [],
    [],
    color="blue",
    marker=">",
    linestyle="dashed",
    markersize=8,
    label=legend_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(
    handles=[green_arrow, blue_arrow],
    bbox_to_anchor=(0.5, 0),
    ncol=2,
    loc="upper center",
)

# Titles and labels
ax1.set_title(title_vehicles)
ax2.set_title(title_services)
ax1.set_xlabel(xlabel)
ax2.set_xlabel(xlabel)
ax1.set_ylabel(ylabel_vehicles)
ax2.set_ylabel(ylabel_services)
fig.suptitle(suptitle)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout(rect=[0, 0, 1, 0.95])
plt.savefig("quiver_43.pdf", bbox_inches="tight")
