# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
# Data for the plot
# Domain: Lawyer Efficiency and Client Satisfaction Across Different Legal Specialties
specialties = ["Corporate Law", "Criminal Law", "Intellectual Property Law"]
efficiency_scores_specialty = [8.7, 7.4, 6.9]
efficiency_bias_specialty = [+2.4, +1.9, +1.2]
satisfaction_scores_specialty = [7.9, 7.0, 6.5]
satisfaction_bias_specialty = [-2.1, -1.8, -1.2]
specialty_labels = ["Efficiency Bias", "Satisfaction Bias"]

departments = ["Litigation", "Compliance", "Patents"]
efficiency_scores_department = [8.5, 7.8, 7.1]
efficiency_bias_department = [+2.7, +2.1, +1.5]
satisfaction_scores_department = [7.8, 7.2, 6.8]
satisfaction_bias_department = [-2.2, -1.9, -1.6]
department_labels = ["Efficiency Bias", "Satisfaction Bias"]

# Text labels
xlabel = "Scores"
ylabel = "Specialties / Departments"
suptitle = "Lawyer Efficiency and Client Satisfaction"
title_specialty = "Specialties"
title_department = "Departments"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(12, 5))

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# First subplot (specialties)
for i in range(len(specialties)):
    # Efficiency bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(efficiency_scores_specialty[i], i + offset * 3 / 2),
        xytext=(
            efficiency_scores_specialty[i] + efficiency_bias_specialty[i],
            i + offset * 3 / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="solid", color="royalblue"),
    )
    ax1.scatter(
        [
            efficiency_scores_specialty[i],
            efficiency_scores_specialty[i] + efficiency_bias_specialty[i],
        ],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{efficiency_bias_specialty[i]:.2f}",
        (
            efficiency_scores_specialty[i] + efficiency_bias_specialty[i],
            i + offset * 1.75,
        ),
        color="royalblue",
        ha="left",
        va="center",
    )

    # Satisfaction bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(satisfaction_scores_specialty[i], i + offset / 2),
        xytext=(
            satisfaction_scores_specialty[i] + satisfaction_bias_specialty[i],
            i + offset / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="dashed", color="darkgrey"),
    )
    ax1.scatter(
        [
            satisfaction_scores_specialty[i],
            satisfaction_scores_specialty[i] + satisfaction_bias_specialty[i],
        ],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{satisfaction_bias_specialty[i]:.2f}",
        (
            satisfaction_scores_specialty[i] + satisfaction_bias_specialty[i],
            i + offset * 0.75,
        ),
        color="darkgrey",
        ha="right",
        va="center",
    )

# Second subplot (departments)
for i in range(len(departments)):
    ax2.annotate(
        "",
        xy=(efficiency_scores_department[i], i + offset * 3 / 2),
        xytext=(
            efficiency_scores_department[i] + efficiency_bias_department[i],
            i + offset * 3 / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="solid", color="royalblue"),
    )
    ax2.scatter(
        [
            efficiency_scores_department[i],
            efficiency_scores_department[i] + efficiency_bias_department[i],
        ],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{efficiency_bias_department[i]:.2f}",
        (
            efficiency_scores_department[i] + efficiency_bias_department[i],
            i + offset * 1.75,
        ),
        color="royalblue",
        ha="left",
        va="center",
    )

    ax2.annotate(
        "",
        xy=(satisfaction_scores_department[i], i + offset / 2),
        xytext=(
            satisfaction_scores_department[i] + satisfaction_bias_department[i],
            i + offset / 2,
        ),
        arrowprops=dict(arrowstyle="<|-", linestyle="dashed", color="darkgrey"),
    )
    ax2.scatter(
        [
            satisfaction_scores_department[i],
            satisfaction_scores_department[i] + satisfaction_bias_department[i],
        ],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{satisfaction_bias_department[i]:.2f}",
        (
            satisfaction_scores_department[i] + satisfaction_bias_department[i],
            i + offset * 0.75,
        ),
        color="darkgrey",
        ha="right",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(specialties))
ax2.set_ylim(0, len(departments))

# Set x-axis limits uniformly
ax1.set_xlim(0, 12)
ax2.set_xlim(0, 12)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(specialties))])
ax1.set_yticklabels(specialties)
ax2.set_yticks([i + offset for i in range(len(departments))])
ax2.set_yticklabels(departments)
ax2.yaxis.tick_right()
ax2.yaxis.set_label_position("right")

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(specialties))], minor=True)
ax2.set_yticks([i for i in range(len(departments))], minor=True)
ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="gray")
ax2.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="gray")

# Add x-axis grid lines and set gap to 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(1))
ax2.xaxis.set_major_locator(plt.MultipleLocator(1))
ax1.grid(axis="x", linestyle="--", linewidth=0.5)
ax2.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
blue_arrow = mlines.Line2D(
    [],
    [],
    color="royalblue",
    marker=">",
    linestyle="solid",
    markersize=8,
    label=specialty_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
grey_arrow = mlines.Line2D(
    [],
    [],
    color="darkgrey",
    marker=">",
    linestyle="dashed",
    markersize=8,
    label=specialty_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[blue_arrow, grey_arrow], bbox_to_anchor=(0.45, 0), ncol=2)

blue_arrow = mlines.Line2D(
    [],
    [],
    color="royalblue",
    marker=">",
    linestyle="solid",
    markersize=8,
    label=department_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
grey_arrow = mlines.Line2D(
    [],
    [],
    color="darkgrey",
    marker=">",
    linestyle="dashed",
    markersize=8,
    label=department_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[blue_arrow, grey_arrow], bbox_to_anchor=(0.85, 0), ncol=2)

# Update labels and titles
ax1.set_xlabel(xlabel)
ax1.set_title(title_specialty)
ax2.set_title(title_department)
fig.suptitle(suptitle)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("quiver_44.pdf", bbox_inches="tight")
