# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
# Data for the plot
regions_1 = ["North America", "Europe", "Australia"]
co2_emissions_1 = [16, 8, 17]  # CO2 Emissions (metric tons per capita)
co2_emissions_change_1 = [-1.2, -0.8, -1.5]  # Change in CO2 emissions (%)
renewable_energy_1 = [20, 30, 25]  # Renewable Energy Usage (%)
renewable_energy_change_1 = [3.5, 2.0, 4.0]  # Change in Renewable Energy Usage (%)

regions_2 = ["South Asia", "Sub-Saharan Africa", "Latin America"]
co2_emissions_2 = [1.8, 0.8, 2.5]  # CO2 Emissions (metric tons per capita)
co2_emissions_change_2 = [0.5, 0.2, 0.7]  # Change in CO2 emissions (%)
renewable_energy_2 = [10, 5, 15]  # Renewable Energy Usage (%)
renewable_energy_change_2 = [1.0, 0.5, 1.2]  # Change in Renewable Energy Usage (%)

ax1_labels = ["Change in CO2 Emissions (%)", "Change in Renewable Energy Usage (%)"]
ax2_labels = ["Change in CO2 Emissions (%)", "Change in Renewable Energy Usage (%)"]

xlabel = "Percentage"
ylabel = "Regions"
title1 = "Developed Regions: Climate Change Data"
title2 = "Developing Regions: Climate Change Data"
suptitle = "Climate Change Data Visualization"
legend_label_co2 = "CO2 Emissions"
legend_label_renewable = "Renewable Energy Usage"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(14, 7))
fig.suptitle(suptitle, fontsize=16)

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5


# Function to plot on the given axis
def plot_climate_data(
    ax,
    regions,
    co2_emissions,
    co2_emissions_change,
    renewable_energy,
    renewable_energy_change,
    labels,
    title,
):
    for i, region in enumerate(regions):
        # CO2 Emissions change line with arrow and grey dots at start and end
        ax.annotate(
            "",
            xy=(co2_emissions[i], i + offset * 3 / 2),
            xytext=(
                co2_emissions[i] + co2_emissions_change[i],
                i + offset * 3 / 2,
            ),
            arrowprops=dict(arrowstyle="<-", color="grey"),
        )
        ax.scatter(
            [
                co2_emissions[i],
                co2_emissions[i] + co2_emissions_change[i],
            ],
            [i + offset * 3 / 2, i + offset * 3 / 2],
            color="black",
            s=25,
        )
        ax.annotate(
            f"{co2_emissions_change[i]:.2f}",
            (co2_emissions[i] + co2_emissions_change[i], i + offset * 1.75),
            color="grey",
            ha="right",
            va="center",
        )

        # Renewable Energy change line with arrow and green dots at start and end
        ax.annotate(
            "",
            xy=(renewable_energy[i], i + offset / 2),
            xytext=(renewable_energy[i] + renewable_energy_change[i], i + offset / 2),
            arrowprops=dict(arrowstyle="<-", color="green"),
        )
        ax.scatter(
            [renewable_energy[i], renewable_energy[i] + renewable_energy_change[i]],
            [i + offset / 2, i + offset / 2],
            color="black",
            s=25,
        )
        ax.annotate(
            f"{renewable_energy_change[i]:.2f}",
            (renewable_energy[i] + renewable_energy_change[i], i + offset * 0.75),
            color="green",
            ha="left",
            va="center",
        )

    # set axis limits
    ax.set_ylim(0, len(regions))
    ax.set_xlim(0, 40)
    # Set y-ticks and labels
    ax.set_yticks([i + offset for i in range(len(regions))])
    ax.set_yticklabels(regions)
    ax.set_xlabel(xlabel)
    ax.set_ylabel(ylabel)
    ax.set_title(title, fontsize=14)

    # Offset grid lines on the y-axis
    ax.set_yticks([i for i in range(len(regions))], minor=True)
    ax.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="grey")

    # add x-axis grid lines
    ax.xaxis.set_major_locator(plt.MultipleLocator(5))
    ax.grid(axis="x", linestyle="--", linewidth=0.5)


# Plot data
plot_climate_data(
    ax1,
    regions_1,
    co2_emissions_1,
    co2_emissions_change_1,
    renewable_energy_1,
    renewable_energy_change_1,
    ax1_labels,
    title1,
)
plot_climate_data(
    ax2,
    regions_2,
    co2_emissions_2,
    co2_emissions_change_2,
    renewable_energy_2,
    renewable_energy_change_2,
    ax2_labels,
    title2,
)

# Set legends
grey_arrow = mlines.Line2D(
    [],
    [],
    color="grey",
    marker=">",
    linestyle="-",
    markersize=8,
    label=legend_label_co2,
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
green_arrow = mlines.Line2D(
    [],
    [],
    color="green",
    marker=">",
    linestyle="-",
    markersize=8,
    label=legend_label_renewable,
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(
    handles=[grey_arrow, green_arrow],
    bbox_to_anchor=(0.5, -0.05),
    loc="upper center",
    ncol=2,
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout(rect=[0, 0.03, 1, 0.95])
plt.savefig("quiver_45.pdf", bbox_inches="tight")
