# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Data for Temperature, Humidity, and Precipitation over 10 months
months = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct"]
temperature = np.random.uniform(20, 30, 10).tolist()
humidity = np.random.uniform(50, 80, 10).tolist()
precipitation = np.random.uniform(100, 200, 10).tolist()

num_vars = len(months)
labels = ["Temperature (°C)", "Humidity (%)", "Precipitation (mm)"]
ticks = [20, 40, 60, 80, 100]
tickslabel = ["20", "40", "60", "80", "100"]

title = "Climate Data Over 10 Months"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Compute angle for each axis
angles = np.linspace(0, 2 * np.pi, num_vars, endpoint=False).tolist()

# The plot is circular, so we need to "complete the loop" and append the start to the end.
temperature += temperature[:1]
humidity += humidity[:1]
precipitation += precipitation[:1]
angles += angles[:1]

# Draw the radar chart
fig, ax = plt.subplots(figsize=(8, 8), subplot_kw=dict(polar=True))

# Temperature
ax.fill(angles, temperature, color="royalblue", alpha=0.25)
ax.plot(
    angles,
    temperature,
    color="royalblue",
    linewidth=2,
    linestyle="-",
    marker="o",
    label=labels[0],
)
ax.scatter(angles[:-1], temperature[:-1], color="royalblue", s=50)

# Humidity
ax.fill(angles, humidity, color="seagreen", alpha=0.25)
ax.plot(
    angles,
    humidity,
    color="seagreen",
    linewidth=2,
    linestyle="--",
    marker="s",
    label=labels[1],
)
ax.scatter(angles[:-1], humidity[:-1], color="seagreen", s=50)

# Precipitation
ax.fill(angles, precipitation, color="darkorange", alpha=0.25)
ax.plot(
    angles,
    precipitation,
    color="darkorange",
    linewidth=2,
    linestyle="-.",
    marker="^",
    label=labels[2],
)
ax.scatter(angles[:-1], precipitation[:-1], color="darkorange", s=50)

# Add labels to the plot
ax.set_yticklabels([])
ax.set_xticks(angles[:-1])
ax.set_xticklabels(months)

# Add grid lines and labels for the concentric circles
ax.set_rgrids(
    ticks,
    labels=tickslabel,
    angle=30,
    color="gray",
    size=10,
)

# Create legend handles manually
legend_elements = [
    Line2D(
        [0],
        [0],
        color="royalblue",
        linewidth=2,
        marker="o",
        linestyle="-",
        markersize=8,
        label=labels[0],
    ),
    Line2D(
        [0],
        [0],
        color="seagreen",
        linewidth=2,
        marker="s",
        linestyle="--",
        markersize=8,
        label=labels[1],
    ),
    Line2D(
        [0],
        [0],
        color="darkorange",
        linewidth=2,
        marker="^",
        linestyle="-.",
        markersize=8,
        label=labels[2],
    ),
]

# Add legend and title
ax.legend(
    handles=legend_elements, loc="upper right", bbox_to_anchor=(1.1, 1.1), frameon=False
)
ax.set_title(title, size=20, color="darkblue")

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("radar_29.pdf", bbox_inches="tight")
