# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Data for Solar and Wind Energy
values_solar = [0.85, 0.4, 0.75, 0.6, 0.9, 0.7, 0.8, 0.6, 0.7, 0.65]
values_wind = [0.4, 0.85, 0.5, 0.55, 0.45, 0.6, 0.7, 0.75, 0.55, 0.8]
labels = [
    "Solar\nEfficiency",
    "Wind\nEfficiency",
    "Hydro\nPotential",
    "Geothermal\nPotential",
    "Carbon\nFootprint",
    "Energy\nCost",
    "Infrastructure\nReadiness",
    "Policy\nSupport",
    "Public\nAcceptance",
    "Technological\nAdvancement",
]
num_vars = len(labels)

# Compute angle for each axis
angles = np.linspace(0, 2 * np.pi, num_vars, endpoint=False).tolist()

# The plot is circular, so we need to "complete the loop" and append the start to the end.
values_solar += values_solar[:1]
values_wind += values_wind[:1]
angles += angles[:1]

# Extracted variables
solar_label = "Solar Energy"
wind_label = "Wind Energy"
xticks = angles[:-1]
xticklabels = labels
yticklabels = []
rgrids = [0.2, 0.4, 0.6, 0.8, 1.0]
rgrid_labels = ["0.2", "0.4", "0.6", "0.8", "1.0"]
title_text = "Energy Source Comparison: Solar vs Wind"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Draw the radar chart
fig, ax = plt.subplots(figsize=(8, 8), subplot_kw=dict(polar=True))

title_size = 18
title_color = "darkgreen"
title_y = 1.1

ax.fill(angles, values_solar, color="seagreen", alpha=0.7)
ax.plot(angles, values_solar, color="seagreen", linewidth=2, label=solar_label)
ax.scatter(angles[:-1], values_solar[:-1], color="seagreen", s=75, zorder=5, marker="^")

ax.fill(angles, values_wind, color="royalblue", alpha=0.7)
ax.plot(angles, values_wind, color="royalblue", linewidth=2, label=wind_label)
ax.scatter(angles[:-1], values_wind[:-1], color="royalblue", s=75, zorder=5, marker="o")

# Add labels to the plot
ax.set_xticks(xticks)
ax.set_xticklabels(xticklabels, size=13)

# Add grid lines and labels for the concentric circles
ax.set_yticklabels(yticklabels)
ax.set_rgrids(rgrids, labels=rgrid_labels, angle=225, color="gray", size=12)

# Create legend handles manually
legend_elements = [
    Line2D(
        [0],
        [0],
        color="seagreen",
        linewidth=2,
        marker="^",
        markersize=10,
        label=solar_label,
    ),
    Line2D(
        [0],
        [0],
        color="royalblue",
        linewidth=2,
        marker="o",
        markersize=10,
        label=wind_label,
    ),
]

# Add legend and title
ax.set_title(title_text, size=title_size, color=title_color, y=title_y)
ax.legend(
    handles=legend_elements,
    loc="lower center",
    bbox_to_anchor=(0.5, -0.2),
    frameon=False,
    ncol=2,
)

ax.xaxis.set_tick_params(pad=20)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("radar_72.pdf", bbox_inches="tight")
