# ===================
# Part 1: Importing Libraries
# ===================

import matplotlib.pyplot as plt
from matplotlib.lines import Line2D

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Data for Country A and Country B
values_country_a = [0.85, 0.45, 0.75, 0.8, 0.7, 0.65, 0.9, 0.85, 0.6, 0.8]
values_country_b = [0.65, 0.75, 0.6, 0.4, 0.3, 0.4, 0.5, 0.6, 0.75, 0.5]
labels = [
    "GDP\nGrowth",
    "Inflation\nRate",
    "Unemployment\nRate",
    "FDI",
    "Trade\nBalance",
    "Public\nDebt",
    "Currency\nStability",
    "Labor\nProductivity",
    "Income\nInequality",
    "Educational\nInvestment",
]
num_vars = len(labels)

# Compute angle for each axis
angles = np.linspace(0, 2 * np.pi, num_vars, endpoint=False).tolist()

# The plot is circular, so we need to "complete the loop" and append the start to the end.
values_country_a += values_country_a[:1]
values_country_b += values_country_b[:1]
angles += angles[:1]

# Extracted variables
country_a_label = "Country A"
country_b_label = "Country B"
xticks = angles[:-1]
xticklabels = labels
yticklabels = []
rgrids = [0.2, 0.4, 0.6, 0.8, 1.0]
rgrid_labels = ["0.2", "0.4", "0.6", "0.8", "1.0"]
title_text = "Economic Indicators Comparison: Country A vs Country B"


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Draw the radar chart
fig, ax = plt.subplots(figsize=(8, 8), subplot_kw=dict(polar=True))

title_size = 18
title_color = "darkblue"
title_y = 1.1

ax.fill(angles, values_country_a, color="skyblue", alpha=0.7)
ax.plot(
    angles,
    values_country_a,
    color="skyblue",
    linewidth=2,
    linestyle="-",
    label=country_a_label,
)
ax.scatter(
    angles[:-1], values_country_a[:-1], color="skyblue", s=75, zorder=5, marker="o"
)
ax.fill(angles, values_country_b, color="lightgreen", alpha=0.7)
ax.plot(
    angles,
    values_country_b,
    color="lightgreen",
    linewidth=2,
    linestyle="--",
    label=country_b_label,
)
ax.scatter(
    angles[:-1], values_country_b[:-1], color="lightgreen", s=75, zorder=5, marker="s"
)

# Add labels to the plot
ax.set_xticks(xticks)
ax.set_xticklabels(xticklabels, size=13)

# Add grid lines and labels for the concentric circles
ax.set_yticklabels(yticklabels)
ax.set_rgrids(
    rgrids,
    labels=rgrid_labels,
    angle=225,
    color="gray",
    size=12,
)

# Create legend handles manually
legend_elements = [
    Line2D(
        [0],
        [0],
        color="skyblue",
        linewidth=2,
        marker="o",
        markersize=10,
        label=country_a_label,
    ),
    Line2D(
        [0],
        [0],
        color="lightgreen",
        linewidth=2,
        marker="s",
        markersize=10,
        label=country_b_label,
    ),
]

# Add legend and title
ax.set_title(
    title_text,
    size=title_size,
    color=title_color,
    y=title_y,
)
ax.legend(
    handles=legend_elements,
    loc="lower center",
    bbox_to_anchor=(0.5, -0.2),
    frameon=False,
    ncol=2,
)
ax.xaxis.set_tick_params(pad=20)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("radar_73.pdf", bbox_inches="tight")
