
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Set a random seed for reproducibility

# Generate 5 equidistant mean values between 0 and -2 (representing reduction in anxiety scores)
mean_values = np.linspace(0, -2, 5)
# Use smaller standard deviations to ensure data falls between -3 and 1
standard_deviations = [0.5] * 5

data_therapy1 = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values, standard_deviations)
]
data_therapy2 = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values, standard_deviations)
]
positions_therapy1 = np.array(range(1, len(data_therapy1) + 1)) - 0.2
positions_therapy2 = np.array(range(1, len(data_therapy2) + 1)) + 0.2
legend_labels = ["Therapy 1", "Therapy 2"]
xlabel = "Session Number"
ylabel = "Reduction in Anxiety Score"
xticks = [1, 2, 3, 4, 5]
xtickslabels = ["Session 1", "Session 2", "Session 3", "Session 4", "Session 5"]


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis with the specified dimensions
fig, ax = plt.subplots(figsize=(8, 7))

# Narrower violin plots
violin_width = 0.35  # set this to a suitable value

# Create the violin plot with adjusted positions

parts_therapy1 = ax.violinplot(
    data_therapy1, positions=positions_therapy1, widths=violin_width, showmeans=False, showmedians=True
)
parts_therapy2 = ax.violinplot(
    data_therapy2, positions=positions_therapy2, widths=violin_width, showmeans=False, showmedians=True
)

# Customizing the colors of the violin plot
for pc in parts_therapy1["bodies"]:
    pc.set_facecolor("#75a3a0")  # Muted teal
    pc.set_edgecolor("black")
    pc.set_alpha(1)

for pc in parts_therapy2["bodies"]:
    pc.set_facecolor("#e89c72")  # Muted coral
    pc.set_edgecolor("black")
    pc.set_alpha(1)

# Customizing the median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = parts_therapy1[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

    vp = parts_therapy2[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

# Customizing the colors of the violin plot and adding statistics
for i in range(len(data_therapy1)):
    # Adding the statistical annotations for data_therapy1
    quartile1, median, quartile3 = np.percentile(data_therapy1[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_therapy1[i][data_therapy1[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_therapy1[i][data_therapy1[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(positions_therapy1[i], quartile1, quartile3, color="black", linestyle="-", lw=4)
    ax.hlines(
        median,
        positions_therapy1[i] - 0.025,
        positions_therapy1[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_therapy1[i], lower_whisker, upper_whisker, color="black", linestyle="-", lw=1
    )

    # Adding the statistical annotations for data_therapy2
    quartile1, median, quartile3 = np.percentile(data_therapy2[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_therapy2[i][data_therapy2[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_therapy2[i][data_therapy2[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(positions_therapy2[i], quartile1, quartile3, color="black", linestyle="-", lw=4)
    ax.hlines(
        median,
        positions_therapy2[i] - 0.025,
        positions_therapy2[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_therapy2[i], lower_whisker, upper_whisker, color="black", linestyle="-", lw=1
    )

# Change the border color to grey
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove small ticks beside the numbers on the x and y axes
ax.tick_params(axis="both", which="both", length=0)

# Adding the corrected legend
ax.legend(
    [parts_therapy1["bodies"][0], parts_therapy2["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.1),
    ncol=2,
)

# Setting the title and labels
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting the x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(
    True, linestyle="-", linewidth=0.7, color="grey", zorder=0
)  # Customize the style of the grid

# ===================
# Part 4: Saving Output
# ===================
# Adjust figure size to match original image's dimensions
fig.set_size_inches(7, 5)  # Width, Height in inches

# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig("violin_23.pdf", bbox_inches="tight")

