# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
import random

np.random.seed(42)

# Generate data for art evaluation scores
data = np.random.beta(a=[5, 20, 10, 18, 12], b=[8, 25, 30, 40, 15], size=(10, 5))
data_memory = np.random.beta(
    a=[7, 30, 35, 15, 45], b=[12, 35, 20, 30, 18], size=(40, 5)
)

# X-axis labels and legend labels
xticklabels = ["A2", "B1", "B2", "C1", "C2"]
legend_labels = ["Teacher-Style", "Standardize"]

# The scaling factor and violin width
scaling_factor = 1
violin_width = 0.5

# Adjust the offsets for 5 groups
offsets = np.linspace(-3, 3, 5)

# Chart labels and title
xlabel = "Evaluation Groups"
ylabel = "Evaluation Scores"
title = "Distribution of Art Evaluation Scores"
supertitle = "Art Evaluation: Teacher-Style vs. Standardize"
legend_title = "Evaluation Conditions"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(8, 6))  # Adjusted figure size

# Define new colors for the artistic theme
colors = ["#6a0dad", "#ff4500"]
legend_colors = ["#ff4500", "#6a0dad"]

# Plot the half-violins with an offset for 5 groups
for i in range(data.shape[1]):
    offset = offsets[i]

    # Plot data without memory (Teacher-Style)
    kde_data = gaussian_kde(data[:, i])
    kde_x = np.linspace(0, 1, 300)
    kde_data_y = kde_data(kde_x)
    kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_data_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        edgecolor="black",
        alpha=0.7,
    )

    # Plot data with memory (Standardize)
    kde_data_memory = gaussian_kde(data_memory[:, i])
    kde_data_memory_y = kde_data_memory(kde_x)
    kde_data_memory_y_scaled = kde_data_memory_y / max(kde_data_memory_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_data_memory_y_scaled * scaling_factor + offset,
        color=colors[1],
        edgecolor="black",
        alpha=0.7,
    )

    # Add yellow stars at the top of each violin plot
    ax.scatter(
        offset,
        random.uniform(0.2, 0.8),
        marker="*",
        color="yellow",
        s=260,
        zorder=3,
        edgecolor="black",
    )

# Set x and y axis labels, limits, and add x-axis tick labels for 5 groups
ax.set_xlim(
    min(offsets) - scaling_factor - violin_width,
    max(offsets) + scaling_factor + violin_width,
)
ax.set_xticks(offsets)  # Set x-ticks to the center of each group
ax.set_xticklabels(xticklabels)  # Label x-ticks accordingly
ax.set_xlabel(xlabel)  # Set x-axis label
ax.set_ylabel(ylabel)  # Set y-axis label

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="black")
    for color in legend_colors
]

ax.legend(handles, legend_labels, loc="upper left", ncol=1, title=legend_title)

# Set the title and supertitle
plt.title(title)
plt.suptitle(supertitle)

# Enhance grid for better readability
ax.grid(True, linestyle="--", alpha=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("violin_33.pdf", bbox_inches="tight")
