# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Sample data to mimic tourism expenditure and duration
expenditure_data = np.random.beta(
    a=[3, 7, 5, 8, 6], b=[10, 12, 15, 20, 10], size=(10, 5)
)
duration_data = np.random.beta(a=[4, 6, 8, 7, 5], b=[18, 22, 20, 16, 24], size=(40, 5))
xticklabels = ["Family", "Couple", "Solo", "Group", "Business"]
legend_labels = ["Expenditure", "Duration"]
# The scaling factor is used to ensure the violins do not overlap
scaling_factor = 1
violin_width = 0.5

# Adjust the offsets for 5 groups instead of 3
offsets = np.linspace(-3, 3, 5)

# Text labels
xlabel = "Tourist Category"
ylabel = "Normalized Density"
title = "Tourism Expenditure and Duration Distribution"
legend_title = "Legend"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(8, 6))

# Define the colors for each group
colors = ["#7fbf7b", "#af8dc3"]
legend_colors = ["#af8dc3", "#7fbf7b"]

# Plot the half-violins with an offset for 5 groups
for i in range(expenditure_data.shape[1]):
    offset = offsets[i]

    # Plot expenditure data
    kde_expenditure = gaussian_kde(expenditure_data[:, i])
    kde_x = np.linspace(0, 1, 300)
    kde_expenditure_y = kde_expenditure(kde_x)
    kde_expenditure_y_scaled = kde_expenditure_y / max(kde_expenditure_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_expenditure_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        edgecolor="black",
        alpha=0.6,
    )

    # Plot duration data
    kde_duration = gaussian_kde(duration_data[:, i])
    kde_duration_y = kde_duration(kde_x)
    kde_duration_y_scaled = kde_duration_y / max(kde_duration_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_duration_y_scaled * scaling_factor + offset,
        color=colors[1],
        edgecolor="black",
        alpha=0.6,
    )

    # Add circle markers at the top of each violin plot
    ax.scatter(
        offset,
        np.random.uniform(0.2, 0.8),
        marker="o",
        color="red",
        s=80,
        zorder=3,
        edgecolor="black",
    )

# Set x and y axis labels, limits, and add x-axis tick labels for 5 groups
ax.set_xlim(
    min(offsets) - scaling_factor - violin_width,
    max(offsets) + scaling_factor + violin_width,
)
ax.set_xticks(offsets)  # Set x-ticks to the center of each group
ax.set_xticklabels(xticklabels)  # Label x-ticks accordingly
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="black", alpha=0.6)
    for color in legend_colors
]
ax.legend(handles, legend_labels, title=legend_title, loc="upper left", ncol=1)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("violin_35.pdf", bbox_inches="tight")
