
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Sample data for temperature in two cities
city1_temps = np.random.normal(25, 5, 365)  # warmer city
city2_temps = np.random.normal(15, 7, 365)  # cooler city

data = [city1_temps, city2_temps]
categories = ["City A", "City B"]
ylabel = "Temperature (°C)"
xlabel = "Cities"
xticks = [1, 2]
ylim = [-10, 40]
title = "Annual Temperature Distribution in Two Cities"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(8, 6))  # Adjusting figure size for better visualization
violin_parts = ax.violinplot(data, showmeans=False, showmedians=True, showextrema=False)

# Customizing the appearance
ax.set_ylabel(ylabel)
ax.set_xlabel(xlabel)
ax.set_xticks(xticks)
ax.set_xticklabels(categories)
ax.set_title(title)
ax.grid(axis="y", alpha=0.6)  # Adding horizontal grid lines
ax.set_ylim(ylim)  # Setting y-axis limits

# Removing the ticks on the x and y axes
ax.tick_params(axis="x", which="both", length=0)  # Remove x-axis ticks
ax.tick_params(axis="y", which="both", length=0)  # Remove y-axis ticks

# Coloring the violins and adding the desired statistical annotations
colors = ["#ff9999", "#66b3ff"]  # Using a different color scheme for weather
for i, (pc, d) in enumerate(zip(violin_parts["bodies"], data)):
    pc.set_facecolor(colors[i])
    pc.set_edgecolor("black")
    pc.set_alpha(0.8)

    # Calculate the quartiles and interquartile range
    quartile1, median, quartile3 = np.percentile(d, [25, 50, 75])
    iqr = quartile3 - quartile1

    # Calculate lower and upper whiskers using 1.5xIQR rule
    lower_whisker = np.min(d[d >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(d[d <= quartile3 + 1.5 * iqr])

    # Placing lines for median, quartiles, and whiskers
    ax.vlines(i + 1, quartile1, quartile3, color="k", linestyle="--", lw=4)
    ax.scatter(i + 1, median, color="w", s=30, zorder=3)
    ax.vlines(i + 1, lower_whisker, upper_whisker, color="k", linestyle="-", lw=1)

# Remove the lines (medians, whiskers, etc.)
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = violin_parts.get(partname)
    if vp:
        vp.set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig("violin_38.pdf", bbox_inches="tight")
