
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(1)
# Generate new sample data for survey of daily water consumption across different age groups
age_group1 = np.clip(np.random.normal(2.5, 0.8, 200), 0, 5)  # Age 0-10
age_group2 = np.clip(np.random.normal(3.0, 0.7, 200), 0, 5)  # Age 11-20
age_group3 = np.clip(np.random.normal(3.5, 0.6, 200), 0, 5)  # Age 21-30
age_group4 = np.clip(np.random.normal(3.2, 0.5, 200), 0, 5)  # Age 31-40
age_group5 = np.clip(np.random.normal(3.0, 0.8, 200), 0, 5)  # Age 41-50

pearson_r = [0.25, 0.28, 0.26, 0.24, 0.22]
eer = [5.5, 6.2, 5.8, 6.1, 5.9]

data = [age_group1, age_group2, age_group3, age_group4, age_group5]
categories = ["Age 0-10", "Age 11-20", "Age 21-30", "Age 31-40", "Age 41-50"]
ylabel = "Daily Water Consumption (Liters)"
ylim = [0, 6]
xlabel = "Age Groups"
textlabels = ["Pearson R", "EER(%)"]
title = "Survey of Daily Water Consumption Across Age Groups"
supertitle = "Water Consumption Survey"
legendlabels = ["Pearson R", "EER(%)"]
text_color_pearson = "darkorange"
text_color_eer = "darkblue"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 7))  # Adjust the figure size to accommodate more violins

# Create violin plots
violin_parts = ax.violinplot(data, showmeans=False, showmedians=True, showextrema=False)

# Customize the appearance
ax.set_ylabel(ylabel)
ax.set_xticks(np.arange(1, len(categories) + 1))  # Adjust the x-ticks to match the number of categories
ax.set_xticklabels(categories)
ax.set_ylim(ylim)  # You may need to adjust this if the data range changes
ax.set_xlabel(xlabel)
ax.set_title(title)
fig.suptitle(supertitle)

# Set violin colors and add statistical annotations
colors = ["#1f77b4", "#ff7f0e", "#2ca02c", "#d62728", "#9467bd"]  # New color scheme

for i, (pc, d) in enumerate(zip(violin_parts['bodies'], data)):
    pc.set_facecolor(colors[i])
    pc.set_edgecolor('black')
    pc.set_alpha(1)

    # Calculate the quartiles and median
    quartile1, median, quartile3 = np.percentile(d, [25, 50, 75])
    iqr = quartile3 - quartile1

    # Calculate whiskers
    lower_whisker = np.min(d[d >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(d[d <= quartile3 + 1.5 * iqr])

    # Annotate statistics
    ax.vlines(i + 1, quartile1, quartile3, color='k', linestyle='-', lw=4)
    ax.scatter(i + 1, median, color='w', s=10, zorder=3)
    ax.vlines(i + 1, lower_whisker, upper_whisker, color='k', linestyle='-', lw=1)
    ax.text(i + 1 + 0.3, np.median(data[i]), f"{median:.2f}", ha="left", va="center", color="black", rotation=45)

    # Annotate with Pearson R and EER values
    ax.text(i + 1, 0.8, f"{pearson_r[i]:.2f}", ha="center", va="center", color=text_color_pearson, fontsize=10)
    ax.text(i + 1, 0.6, f"{eer[i]:.2f}", ha="center", va="center", color=text_color_eer, fontsize=10)

ax.text(5.6, 0.8, textlabels[0], ha="left", va="center", color=text_color_pearson, fontsize=10)
ax.text(5.6, 0.6, textlabels[1], ha="left", va="center", color=text_color_eer, fontsize=10)

# Make the other parts of the violin plots invisible
for partname in ('cbars', 'cmins', 'cmaxes', 'cmedians'):
    vp = violin_parts.get(partname)
    if vp:
        vp.set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig("violin_43.pdf", bbox_inches="tight")
