
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Generates new data for sports performance metrics
endurance_data = np.clip(np.random.normal(0.75, 0.1, 200), 0, 1)
agility_data = np.clip(np.random.normal(0.85, 0.05, 200), 0, 1)
strength_data = np.clip(np.random.normal(0.65, 0.12, 200), 0, 1)
speed_data = np.clip(np.random.normal(0.70, 0.08, 200), 0, 1)
overall_data = np.clip(np.random.normal(0.80, 0.1, 200), 0, 1)

correlation_coeff = [0.2, 0.22, 0.18, 0.21, 0.24]
performance_index = [45.5, 42.3, 40.8, 44.1, 46.0]

data = [endurance_data, agility_data, strength_data, speed_data, overall_data]
categories = ["Endurance", "Agility", "Strength", "Speed", "Overall"]
ylabel = "Performance Metric"
ylim = [0, 1.1]
xlabel = "Training Regime"
textlabels = ["Correlation Coeff", "Performance Index(%)"]
title = "Athlete Performance Across Training Regimes"
supertitle = "Sports Performance Metrics"
legendlabels = ["Median", "Quartiles", "Whiskers"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 7))

# Create violin plots
violin_parts = ax.violinplot(data, showmeans=False, showmedians=True, showextrema=False)

# Customize the appearance
ax.set_ylabel(ylabel)
ax.set_xticks(np.arange(1, len(categories) + 1))
ax.set_xticklabels(categories)
ax.set_ylim(ylim)
ax.set_xlabel(xlabel)
ax.set_title(title)
fig.suptitle(supertitle)

# Set violin colors and add statistical annotations
colors = ["#ff9999", "#66b3ff", "#99ff99", "#ffcc99", "#c2c2f0"]
for i, (pc, d) in enumerate(zip(violin_parts["bodies"], data)):
    pc.set_facecolor(colors[i])
    pc.set_edgecolor("black")
    pc.set_alpha(1)

    # Calculate the quartiles and median
    quartile1, median, quartile3 = np.percentile(d, [25, 50, 75])
    iqr = quartile3 - quartile1

    # Calculate whiskers
    lower_whisker = np.min(d[d >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(d[d <= quartile3 + 1.5 * iqr])

    # Annotate statistics
    ax.vlines(i + 1, quartile1, quartile3, color="k", linestyle="-", lw=4)
    ax.scatter(i + 1, median, color="red", s=10, zorder=3)
    ax.vlines(i + 1, lower_whisker, upper_whisker, color="k", linestyle="-", lw=1)
    ax.text(
        i + 1 + 0.3,
        np.median(data[i]),
        f"{median:.2f}",
        ha="left",
        va="center",
        color="black",
        rotation=45,
    )

    # Annotate with Correlation Coeff and Performance Index values
    ax.text(
        i + 1,
        0.14,
        f"{correlation_coeff[i]:.2f}",
        ha="center",
        va="center",
        color="green",
        fontsize=10,
    )
    ax.text(
        i + 1,
        0.08,
        f"{performance_index[i]:.2f}",
        ha="center",
        va="center",
        color="blue",
        fontsize=10,
    )

ax.text(5.6, 0.14, textlabels[0], ha="left", va="center", color="green", fontsize=10)
ax.text(5.6, 0.08, textlabels[1], ha="left", va="center", color="blue", fontsize=10)

# Make the other parts of the violin plots invisible
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = violin_parts.get(partname)
    if vp:
        vp.set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()

# Display the plot
plt.savefig("violin_44.pdf", bbox_inches="tight")

