
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate mean values for battery life (in hours)
mean_values_model_x = np.linspace(15, 25, 5)
mean_values_model_y = np.linspace(20, 30, 5)

# Standard deviations
standard_deviations = [5] * 5

# Generate synthetic data for Smartphone Model X
data_model_x = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values_model_x, standard_deviations)
]

# Generate synthetic data for Smartphone Model Y
data_model_y = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values_model_y, standard_deviations)
]

# Set positions for violins
positions_model_x = np.array(range(1, len(data_model_x) + 1)) - 0.2
positions_model_y = np.array(range(1, len(data_model_y) + 1)) + 0.2

# Labels and ticks
legend_labels = ["Smartphone Model X", "Smartphone Model Y"]
xlabel = "Usage Patterns"
ylabel = "Battery Life (Hours)"
xticks = [1, 2, 3, 4, 5]
xtickslabels = ["Browsing", "Streaming", "Gaming", "Standby", "Video Call"]
title = "Battery Life Comparison under Different Usage Patterns"
suitxt = "Smartphone Battery Life Analysis"
# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(8, 7))

violin_width = 0.35

# Create violin plots
parts_model_x = ax.violinplot(
    data_model_x,
    positions=positions_model_x,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)
parts_model_y = ax.violinplot(
    data_model_y,
    positions=positions_model_y,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)

# Customizing colors
for pc in parts_model_x["bodies"]:
    pc.set_facecolor("teal")
    pc.set_edgecolor("black")
    pc.set_alpha(1)

for pc in parts_model_y["bodies"]:
    pc.set_facecolor("orange")
    pc.set_edgecolor("black")
    pc.set_alpha(1)

# Customizing median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp_model_x = parts_model_x[partname]
    vp_model_y = parts_model_y[partname]
    vp_model_x.set_edgecolor("black")
    vp_model_y.set_edgecolor("black")
    vp_model_x.set_linewidth(1)
    vp_model_y.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp_model_x.set_visible(False)  # Hide caps
        vp_model_y.set_visible(False)  # Hide caps

# Adding statistical annotations
for i in range(len(data_model_x)):
    quartile1, median, quartile3 = np.percentile(data_model_x[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_model_x[i][data_model_x[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_model_x[i][data_model_x[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_model_x[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_model_x[i] - 0.025,
        positions_model_x[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_model_x[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

    quartile1, median, quartile3 = np.percentile(data_model_y[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_model_y[i][data_model_y[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_model_y[i][data_model_y[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_model_y[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_model_y[i] - 0.025,
        positions_model_y[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_model_y[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

# Customize borders
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove ticks
ax.tick_params(axis="both", which="both", length=0)

# Adding legend
ax.legend(
    [parts_model_x["bodies"][0], parts_model_y["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.0),
    ncol=2,
)

# Setting labels
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(True, linestyle="-", linewidth=0.7, color="grey", zorder=0)

# Adding title
plt.title(title)
plt.suptitle(suitxt, y=0.98, fontsize=13)

# ===================
# Part 4: Saving Output
# ===================
fig.set_size_inches(7, 5)

plt.tight_layout()
plt.savefig("violin_58.pdf", bbox_inches="tight")
