
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Generate synthetic data for technology domain
data_office = np.random.beta(a=[3, 6, 4], b=[4, 3, 2], size=(100, 3))
data_remote = np.random.beta(a=[5, 7, 3], b=[2, 2, 4], size=(100, 3))

categories = ["Code Quality", "Team Collaboration", "Feature Delivery Speed"]
violin_width = 0.02

# Axes Limits and Labels
xlabel_value = "Performance Metrics"
ylabel_value = "Scores"
labels = ["Office Environment", "Remote Environment"]
title = "Performance Comparison in Technology Workspaces"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(8, 7))  # Larger figure for improved readability

# Define the categories and the colors for each group
colors = ["skyblue", "lightgreen"]

# The scaling factor is used to ensure the violins do not overlap
scaling_factor = 1

# Define offset to separate the half violin plots in the single Axes object
offsets = [-0.05, 0, 0.05]

# Plot the half-violins with an offset
for i, category in enumerate(categories):
    offset = offsets[i]

    # Plot data for office environment
    kde_data_office = gaussian_kde(data_office[:, i])
    kde_x = np.linspace(0, 1, 300)
    kde_data_office_y = kde_data_office(kde_x)
    kde_data_office_y_scaled = (
        kde_data_office_y / max(kde_data_office_y) * violin_width
    )
    ax.fill_betweenx(
        kde_x,
        kde_data_office_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        alpha=0.6,
        edgecolor="#505050"
    )

    # Plot data for remote environment
    kde_data_remote = gaussian_kde(data_remote[:, i])
    kde_data_remote_y = kde_data_remote(kde_x)
    kde_data_remote_y_scaled = (
        kde_data_remote_y / max(kde_data_remote_y) * violin_width
    )
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_data_remote_y_scaled * scaling_factor + offset,
        color=colors[1],
        alpha=0.6,
        edgecolor="#505050",
    )
    ax.text(
        offset, -0.1, category, ha="center", va="top"
    )  # Add the category label below the violin plot

# Set x and y axis labels and limits
ax.set_xlim(
    min(offsets) - scaling_factor * violin_width - 0.01,
    max(offsets) + scaling_factor * violin_width + 0.01,
)
y_margin = 0.01  # Adding margin at top and bottom of the y-axis
y_min, y_max = ax.get_ylim()
ax.set_ylim(y_min - y_margin, y_max + y_margin)
ax.set_ylabel(ylabel_value)
ax.set_xticks([])  # Remove x-ticks as they are not meaningful here

# Add a meaningful chart title and grid
ax.set_title(title)
ax.grid(True, linestyle='--', alpha=0.6)

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="#505050") for color in colors
]
ax.legend(handles, labels, loc="upper left", bbox_to_anchor=(0.7, 1), ncol=1)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better fit and save the plot
plt.tight_layout()
plt.savefig("violin_62.pdf", bbox_inches="tight")
