
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np; np.random.seed(0)
# Simulated rainfall data in mm for two cities across six months
data_6_months_city_A = np.random.normal(loc=[50, 70, 60, 40, 80, 90], scale=[10, 15, 10, 20, 10, 15], size=(100, 6))
data_6_months_city_B = np.random.normal(loc=[60, 80, 50, 70, 90, 100], scale=[15, 10, 15, 10, 20, 10], size=(100, 6))
ylabel="Average Monthly Rainfall (mm)"
ylim=[0, 120]
violin_width = 0.4
scaling_factor = 1
kde_x = np.linspace(0, 120, 300)

# Offsets for months
offsets_6_months = np.linspace(-3, 3, 6)
labels=["City A", "City B"]
title="Average Monthly Rainfall Distribution for Cities"
legend_labels = ["City A", "City B"]
xticklabels=["Jan", "Feb", "Mar", "Apr", "May", "Jun"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure
fig, ax = plt.subplots(figsize=(12, 7))

# Define the colors for each month
colors = ["#1f77b4", "#aec7e8", "#ffbb78", "#2ca02c", "#98df8a", "#d62728"]

# Function to plot half violins for rainfall data
def plot_half_violins(ax, data_city_A, data_city_B, offsets, colors, labels, title, xticklabels):
    # Plot the half-violins with an offset for months
    for i in range(data_city_A.shape[1]):
        offset = offsets[i]

        # Plot data for City A
        kde_data_A = gaussian_kde(data_city_A[:, i])
        kde_x = np.linspace(0, 120, 300)
        kde_data_A_y = kde_data_A(kde_x)
        kde_data_A_y_scaled = kde_data_A_y / max(kde_data_A_y) * violin_width
        ax.fill_betweenx(
            kde_x,
            kde_data_A_y_scaled * scaling_factor + offset,
            offset,
            color=colors[i],
            edgecolor="black",
            alpha=0.6
        )

        # Plot data for City B
        kde_data_B = gaussian_kde(data_city_B[:, i])
        kde_data_B_y = kde_data_B(kde_x)
        kde_data_B_y_scaled = kde_data_B_y / max(kde_data_B_y) * violin_width
        ax.fill_betweenx(
            kde_x,
            offset,
            -kde_data_B_y_scaled * scaling_factor + offset,
            color=colors[i],
            edgecolor="black",
            alpha=0.3
        )

    # Set x and y axis labels, limits, and add x-axis tick labels for months
    ax.set_xlim(
        min(offsets) - scaling_factor - violin_width,
        max(offsets) + scaling_factor + violin_width,
    )
    ax.set_ylim(ylim)  # Set y-axis limits to 0-120 mm
    ax.set_ylabel(ylabel)
    ax.set_xticks(offsets)  # Set x-ticks to the center of each month
    ax.set_xticklabels(xticklabels)  # Label x-ticks accordingly
    ax.title.set_text(title)

# Plot the violins
plot_half_violins(
    ax,
    data_6_months_city_A,
    data_6_months_city_B,
    offsets_6_months,
    colors,
    labels,
    title,
    xticklabels
)

# Add a legend for the entire figure
handles = [
    plt.Line2D(
        [0], [0], marker="o", color="w", markerfacecolor="black", markersize=10, alpha=0.6
    ),
    plt.Line2D(
        [0], [0], marker="o", color="w", markerfacecolor="black", markersize=10, alpha=0.3
    ),
]

fig.legend(handles, legend_labels, loc="lower center", bbox_to_anchor=(0.5, -0.1), ncol=2)

# ===================
# Part 4: Saving Output
# ===================
# Tighten the layout and show the combined plot
plt.tight_layout()

# Display the plot
plt.savefig("violin_68.pdf", bbox_inches="tight")
