# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generate plausible data for environmental awareness and actions preferences
data = np.random.beta(a=[10, 20, 15, 25, 30], b=[25, 15, 20, 20, 25], size=(10, 5))
data_memory = np.random.beta(
    a=[20, 25, 30, 15, 35], b=[30, 20, 25, 25, 30], size=(40, 5)
)

xticklabels = [
    "Recycling",
    "Composting",
    "Carpooling",
    "Public Transport",
    "Renewable Energy",
]
legend_labels = ["Current Engagement", "Historical Trends"]
xlabel = "Environmental Activities"
ylabel = "Preference Distribution"
title = "Environmental Awareness and Actions Preferences"
legend_title = "Data Source"
scaling_factor = 1
violin_width = 0.5

# Adjust the offsets for 5 groups
offsets = np.linspace(-3, 3, 5)

suptitle = "Violin Plot of Environmental Activities"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(10, 7))

# Define the colors for each group
colors = ["#66c2a5", "#3288bd"]
legend_colors = ["#3288bd", "#66c2a5"]

# Plot the half-violins with an offset for 5 groups
for i in range(data.shape[1]):
    offset = offsets[i]

    # Plot data without memory
    kde_data = gaussian_kde(data[:, i])
    kde_x = np.linspace(0, 1, 300)
    kde_data_y = kde_data(kde_x)
    kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_data_y_scaled + offset,
        offset,
        color=colors[0],
        edgecolor="black",
        alpha=0.6,
    )

    # Plot data with memory
    kde_data_memory = gaussian_kde(data_memory[:, i])
    kde_data_memory_y = kde_data_memory(kde_x)
    kde_data_memory_y_scaled = kde_data_memory_y / max(kde_data_memory_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_data_memory_y_scaled + offset,
        color=colors[1],
        edgecolor="black",
        alpha=0.6,
    )

    # Add stylish markers at the top of each violin plot
    ax.scatter(
        offset,
        np.mean(kde_x),
        marker="^",
        color="orange",
        s=120,
        zorder=3,
        edgecolor="black",
    )

# Set x and y axis labels, limits, and add x-axis tick labels for 5 groups
ax.set_xlim(
    min(offsets) - scaling_factor - violin_width,
    max(offsets) + scaling_factor + violin_width,
)
ax.set_xticks(offsets)
ax.set_xticklabels(xticklabels, fontsize=12, fontweight="bold")
ax.set_xlabel(xlabel, fontsize=15, fontweight="bold")
ax.set_ylabel(ylabel, fontsize=15, fontweight="bold")

# Set the title
ax.set_title(title, fontsize=16, fontweight="bold", pad=20)

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="black")
    for color in legend_colors
]
ax.legend(
    handles,
    legend_labels,
    loc="upper left",
    fontsize=12,
    title=legend_title,
    title_fontsize="13",
)

# Add a grid for better readability
ax.grid(True, which="both", linestyle="--", linewidth=0.5)

# Adding suptitle
fig.suptitle(suptitle, fontsize=18, fontweight="bold")

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout(rect=[0, 0, 1, 0.95])
plt.savefig("violin_76.pdf", bbox_inches="tight")
