from inspect import isfunction
import math
import torch
import torch.nn.functional as F
from torch import nn, einsum
from einops import rearrange, repeat
import numpy as np 

# from ldm.modules.diffusionmodules.util import checkpoint, FourierEmbedder
from torch.utils import checkpoint

def exists(val):
    return val is not None


def uniq(arr):
    return{el: True for el in arr}.keys()


def default(val, d):
    if exists(val):
        return val
    return d() if isfunction(d) else d


def max_neg_value(t):
    return -torch.finfo(t.dtype).max


def init_(tensor):
    dim = tensor.shape[-1]
    std = 1 / math.sqrt(dim)
    tensor.uniform_(-std, std)
    return tensor


# feedforward
class GEGLU(nn.Module):
    def __init__(self, dim_in, dim_out):
        super().__init__()
        self.proj = nn.Linear(dim_in, dim_out * 2)

    def forward(self, x):
        x, gate = self.proj(x).chunk(2, dim=-1)
        return x * F.gelu(gate)


class FeedForward(nn.Module):
    def __init__(self, dim, dim_out=None, mult=4, glu=False, dropout=0.):
        super().__init__()
        inner_dim = int(dim * mult)
        dim_out = default(dim_out, dim)
        project_in = nn.Sequential(
            nn.Linear(dim, inner_dim),
            nn.GELU()
        ) if not glu else GEGLU(dim, inner_dim)

        self.net = nn.Sequential(
            project_in,
            nn.Dropout(dropout),
            nn.Linear(inner_dim, dim_out)
        )

    def forward(self, x):
        return self.net(x)


def zero_module(module):
    """
    Zero out the parameters of a module and return it.
    """
    for p in module.parameters():
        p.detach().zero_()
    return module


def Normalize(in_channels):
    return torch.nn.GroupNorm(num_groups=32, num_channels=in_channels, eps=1e-6, affine=True)


class LinearAttention(nn.Module):
    def __init__(self, dim, heads=4, dim_head=32):
        super().__init__()
        self.heads = heads
        hidden_dim = dim_head * heads
        self.to_qkv = nn.Conv2d(dim, hidden_dim * 3, 1, bias = False)
        self.to_out = nn.Conv2d(hidden_dim, dim, 1)

    def forward(self, x):
        b, c, h, w = x.shape
        qkv = self.to_qkv(x)
        q, k, v = rearrange(qkv, 'b (qkv heads c) h w -> qkv b heads c (h w)', heads = self.heads, qkv=3)
        k = k.softmax(dim=-1)  
        context = torch.einsum('bhdn,bhen->bhde', k, v)
        out = torch.einsum('bhde,bhdn->bhen', context, q)
        out = rearrange(out, 'b heads c (h w) -> b (heads c) h w', heads=self.heads, h=h, w=w)
        return self.to_out(out)





class CrossAttention(nn.Module):
    def __init__(self, query_dim, key_dim, value_dim, heads=8, dim_head=64, dropout=0):
        super().__init__()
        inner_dim = dim_head * heads
        self.scale = dim_head ** -0.5
        self.heads = heads


        self.to_q = nn.Linear(query_dim, inner_dim, bias=False)
        self.to_k = nn.Linear(key_dim, inner_dim, bias=False)
        self.to_v = nn.Linear(value_dim, inner_dim, bias=False)


        self.to_out = nn.Sequential( nn.Linear(inner_dim, query_dim), nn.Dropout(dropout) )


    def fill_inf_from_mask(self, sim, mask):
        if mask is not None:
            B,M = mask.shape
            mask = mask.unsqueeze(1).repeat(1,self.heads,1).reshape(B*self.heads,1,-1)
            max_neg_value = -torch.finfo(sim.dtype).max
            sim.masked_fill_(~mask, max_neg_value)
        return sim 


    def forward(self, x, key, value, mask=None):

        q = self.to_q(x)     # B*N*(H*C)
        k = self.to_k(key)   # B*M*(H*C)
        v = self.to_v(value) # B*M*(H*C)
   
        B, N, HC = q.shape 
        _, M, _ = key.shape
        H = self.heads
        C = HC // H 

        q = q.view(B,N,H,C).permute(0,2,1,3).reshape(B*H,N,C) # (B*H)*N*C
        k = k.view(B,M,H,C).permute(0,2,1,3).reshape(B*H,M,C) # (B*H)*M*C
        v = v.view(B,M,H,C).permute(0,2,1,3).reshape(B*H,M,C) # (B*H)*M*C

        sim = torch.einsum('b i d, b j d -> b i j', q, k) * self.scale # (B*H)*N*M
        self.fill_inf_from_mask(sim, mask)
        attn = sim.softmax(dim=-1) # (B*H)*N*M

        out = torch.einsum('b i j, b j d -> b i d', attn, v) # (B*H)*N*C
        out = out.view(B,H,N,C).permute(0,2,1,3).reshape(B,N,(H*C)) # B*N*(H*C)

        return self.to_out(out)




class SelfAttention(nn.Module):
    def __init__(self, query_dim, heads=8, dim_head=64, dropout=0.):
        super().__init__()
        inner_dim = dim_head * heads
        self.scale = dim_head ** -0.5
        self.heads = heads

        self.to_q = nn.Linear(query_dim, inner_dim, bias=False)
        self.to_k = nn.Linear(query_dim, inner_dim, bias=False)
        self.to_v = nn.Linear(query_dim, inner_dim, bias=False)

        self.to_out = nn.Sequential(nn.Linear(inner_dim, query_dim), nn.Dropout(dropout) )

    def forward(self, x):
        q = self.to_q(x) # B*N*(H*C)
        k = self.to_k(x) # B*N*(H*C)
        v = self.to_v(x) # B*N*(H*C)

        B, N, HC = q.shape 
        H = self.heads
        C = HC // H 

        q = q.view(B,N,H,C).permute(0,2,1,3).reshape(B*H,N,C) # (B*H)*N*C
        k = k.view(B,N,H,C).permute(0,2,1,3).reshape(B*H,N,C) # (B*H)*N*C
        v = v.view(B,N,H,C).permute(0,2,1,3).reshape(B*H,N,C) # (B*H)*N*C

        sim = torch.einsum('b i c, b j c -> b i j', q, k) * self.scale  # (B*H)*N*N
        attn = sim.softmax(dim=-1) # (B*H)*N*N

        out = torch.einsum('b i j, b j c -> b i c', attn, v) # (B*H)*N*C
        out = out.view(B,H,N,C).permute(0,2,1,3).reshape(B,N,(H*C)) # B*N*(H*C)

        return self.to_out(out)



class GatedCrossAttentionDense(nn.Module):
    def __init__(self, query_dim, key_dim, value_dim, n_heads, d_head):
        super().__init__()
        
        self.attn = CrossAttention(query_dim=query_dim, key_dim=key_dim, value_dim=value_dim, heads=n_heads, dim_head=d_head) 
        self.ff = FeedForward(query_dim, glu=True)

        self.norm1 = nn.LayerNorm(query_dim)
        self.norm2 = nn.LayerNorm(query_dim)

        self.register_parameter('alpha_attn', nn.Parameter(torch.tensor(0.)) )
        self.register_parameter('alpha_dense', nn.Parameter(torch.tensor(0.)) )

        # this can be useful: we can externally change magnitude of tanh(alpha)
        # for example, when it is set to 0, then the entire model is same as original one 
        self.scale = 1  

    def forward(self, x, objs):

        x = x + self.scale*torch.tanh(self.alpha_attn) * self.attn( self.norm1(x), objs, objs)  
        x = x + self.scale*torch.tanh(self.alpha_dense) * self.ff( self.norm2(x) ) 
        
        return x 


class GatedSelfAttentionDense(nn.Module):
    def __init__(self, query_dim, context_dim,  n_heads, d_head):
        super().__init__()
        
        # we need a linear projection since we need cat visual feature and obj feature
        self.linear = nn.Linear(context_dim, query_dim)

        self.attn = SelfAttention(query_dim=query_dim, heads=n_heads, dim_head=d_head)
        self.ff = FeedForward(query_dim, glu=True)

        self.norm1 = nn.LayerNorm(query_dim)
        self.norm2 = nn.LayerNorm(query_dim)

        self.register_parameter('alpha_attn', nn.Parameter(torch.tensor(0.)) )
        self.register_parameter('alpha_dense', nn.Parameter(torch.tensor(0.)) )

        # this can be useful: we can externally change magnitude of tanh(alpha)
        # for example, when it is set to 0, then the entire model is same as original one 
        self.scale = 1  


    def forward(self, x, objs):

        N_visual = x.shape[1]
        objs = self.linear(objs)
        
        # print("x.shape:", x.shape, "objs.shape:", objs.shape)
        # x.shape: torch.Size([3, 4096, 320]) objs.shape: torch.Size([3, 30, 320])
        
        
        x = x + self.scale*torch.tanh(self.alpha_attn) * self.attn(  self.norm1(torch.cat([x,objs],dim=1))  )[:,0:N_visual,:]
        x = x + self.scale*torch.tanh(self.alpha_dense) * self.ff( self.norm2(x) )  
        
        return x 






class GatedSelfAttentionDense2(nn.Module):
    def __init__(self, query_dim, context_dim,  n_heads, d_head):
        super().__init__()
        
        # we need a linear projection since we need cat visual feature and obj feature
        self.linear = nn.Linear(context_dim, query_dim)

        self.attn = SelfAttention(query_dim=query_dim, heads=n_heads, dim_head=d_head)
        self.ff = FeedForward(query_dim, glu=True)

        self.norm1 = nn.LayerNorm(query_dim)
        self.norm2 = nn.LayerNorm(query_dim)

        self.register_parameter('alpha_attn', nn.Parameter(torch.tensor(0.)) )
        self.register_parameter('alpha_dense', nn.Parameter(torch.tensor(0.)) )

        # this can be useful: we can externally change magnitude of tanh(alpha)
        # for example, when it is set to 0, then the entire model is same as original one 
        self.scale = 1  


    def forward(self, x, objs):

        B, N_visual, _ = x.shape
        B, N_ground, _ = objs.shape

        objs = self.linear(objs)
        
        # sanity check 
        size_v = math.sqrt(N_visual)
        size_g = math.sqrt(N_ground)
        assert int(size_v) == size_v, "Visual tokens must be square rootable"
        assert int(size_g) == size_g, "Grounding tokens must be square rootable"
        size_v = int(size_v)
        size_g = int(size_g)

        # select grounding token and resize it to visual token size as residual 
        out = self.attn(  self.norm1(torch.cat([x,objs],dim=1))  )[:,N_visual:,:]
        out = out.permute(0,2,1).reshape( B,-1,size_g,size_g )
        out = torch.nn.functional.interpolate(out, (size_v,size_v), mode='bicubic')
        residual = out.reshape(B,-1,N_visual).permute(0,2,1)
        
        # add residual to visual feature 
        x = x + self.scale*torch.tanh(self.alpha_attn) * residual
        x = x + self.scale*torch.tanh(self.alpha_dense) * self.ff( self.norm2(x) )  
        
        return x 




##### video ldm blocks #####
from einops import rearrange
from einops.layers.torch import Rearrange
import torch.nn.functional as F

class PositionalEncoding(nn.Module):
    def __init__(self, dim, max_pos=512):
        super().__init__()

        pos = torch.arange(max_pos)

        freq = torch.arange(dim//2) / dim
        freq = (freq * torch.tensor(10000).log()).exp()

        x = rearrange(pos, 'L -> L 1') / freq
        x = rearrange(x, 'L d -> L d 1')

        pe = torch.cat((x.sin(), x.cos()), dim=-1)
        self.pe = rearrange(pe, 'L d sc -> L (d sc)')

        #self.dummy = nn.Parameter(torch.rand(1), requires_grad=False)

    def forward(self, length):
        enc = self.pe[:length]
        #enc = enc.to(self.dummy.device)
        enc = enc.cuda()
        return enc
    
    

class Conv3DLayer(nn.Module):
    def __init__(self, in_dim, out_dim, n_frames):
        super().__init__()

        #self.to_3d = Rearrange('(b t) c h w -> b c t h w', t=n_frames)
        #self.to_2d = Rearrange('b c t h w -> (b t) c h w')

        k, p = (3, 1, 1), (1, 0, 0)
        self.block1 = nn.Sequential(
            nn.GroupNorm(32, in_dim),
            nn.SiLU(),
            nn.Conv3d(in_dim, out_dim, kernel_size=k, stride=1, padding=p)
        )
        self.block2 = nn.Sequential(
            nn.GroupNorm(32, out_dim),
            nn.SiLU(),
            nn.Conv3d(out_dim, out_dim, kernel_size=k, stride=1, padding=p)
        )

        #self.alpha = nn.Parameter(torch.ones(1))
        self.alpha = nn.Parameter(torch.zeros(1), requires_grad=True)
        #self.alpha = torch.Tensor([0.5]).cuda()

    def forward(self, x, n_frames):
        #h = self.to_3d(x)
        h = rearrange(x, '(b t) c h w -> b c t h w', t=n_frames)

        h = self.block1(h)
        h = self.block2(h)

        #h = self.to_2d(h)
        h = rearrange(h, 'b c t h w -> (b t) c h w')

        with torch.no_grad():
            self.alpha.clamp_(0, 1)

        out = self.alpha * x + (1 - self.alpha) * h
        return out



class TemporalAttentionLayer(nn.Module):
    def __init__(self, dim, n_frames, n_heads=8, kv_dim=None):
        super().__init__()
        self.n_frames = n_frames
        self.n_heads = n_heads

        self.pos_enc = PositionalEncoding(dim)

        head_dim = dim // n_heads
        proj_dim = head_dim * n_heads
        self.q_proj = nn.Linear(dim, proj_dim, bias=False)

        kv_dim = kv_dim or dim
        self.k_proj = nn.Linear(kv_dim, proj_dim, bias=False)
        self.v_proj = nn.Linear(kv_dim, proj_dim, bias=False)
        self.o_proj = nn.Linear(proj_dim, dim, bias=False)

        #self.alpha = nn.Parameter(torch.ones(1))
        self.alpha = nn.Parameter(torch.zeros(1), requires_grad=True)
        #self.alpha = torch.Tensor([0.5]).cuda()
        
    def forward(self, q, kv=None, mask=None, n_frames=None):
        skip = q

        bt, c, h, w = q.shape
        #q = rearrange(q, '(b t) c h w -> b (h w) t c', t=self.n_frames)
        q = rearrange(q, '(b t) c h w -> b (h w) t c', t=n_frames)
        
        #q = q + self.pos_enc(self.n_frames)
        q = q + self.pos_enc(n_frames)

        #kv = kv[::self.n_frames] if kv is not None else q
        kv = kv[::n_frames] if kv is not None else q
        
        q = self.q_proj(q)
        k = self.k_proj(kv)
        v = self.v_proj(kv)

        #print("n_frames:", n_frames)
        
        q = rearrange(q, 'b hw t (heads d) -> b hw heads t d', heads=self.n_heads)
        k = rearrange(k, 'b s (heads d) -> b heads s d', heads=self.n_heads).unsqueeze(1)
        v = rearrange(v, 'b s (heads d) -> b heads s d', heads=self.n_heads).unsqueeze(1)
        
        #print("##### q.shape", q.shape)
        #print("##### k.shape", k.shape)
        #print("##### v.shape", v.shape)
        #if mask is not None:
        #    print("##### mask.shape", mask.shape)
        
        ##### q.shape torch.Size([1, 4096, 8, 4, 40])
        ##### k.shape torch.Size([1, 8, 77, 40])
        ##### v.shape torch.Size([1, 8, 77, 40])
        
        
        out = F.scaled_dot_product_attention(q, k, v, mask)
        out = rearrange(out, 'b hw heads t d -> b hw t (heads d)')
        out = self.o_proj(out)

        out = rearrange(out, 'b (h w) t c -> (b t) c h w', h=h, w=w)

        with torch.no_grad():
            self.alpha.clamp_(0, 1)

        out = self.alpha * skip + (1 - self.alpha) * out
        return out
    
    
    

class BasicTransformerBlock(nn.Module):
    def __init__(self, query_dim, key_dim, value_dim, n_heads, d_head, fuser_type, n_frames = 8, use_checkpoint=True, use_videoldm=True):
        super().__init__()
        self.attn1 = SelfAttention(query_dim=query_dim, heads=n_heads, dim_head=d_head)  
        self.ff = FeedForward(query_dim, glu=True)
        self.attn2 = CrossAttention(query_dim=query_dim, key_dim=key_dim, value_dim=value_dim, heads=n_heads, dim_head=d_head)  
        self.norm1 = nn.LayerNorm(query_dim)
        self.norm2 = nn.LayerNorm(query_dim)
        self.norm3 = nn.LayerNorm(query_dim)
        
        
        
        self.use_checkpoint = use_checkpoint
        self.n_frames = n_frames

        if fuser_type == "gatedSA": #####
            # note key_dim here actually is context_dim
            self.fuser = GatedSelfAttentionDense(query_dim, key_dim, n_heads, d_head) 
            self.enable_fuser = True 
        elif fuser_type == "gatedSA2":
            # note key_dim here actually is context_dim
            self.fuser = GatedSelfAttentionDense2(query_dim, key_dim, n_heads, d_head) 
            self.enable_fuser = True
        elif fuser_type == "gatedCA":
            self.fuser = GatedCrossAttentionDense(query_dim, key_dim, value_dim, n_heads, d_head) 
            self.enable_fuser = True
        elif fuser_type is None:
            self.enable_fuser = False
        else:
            assert False 
        
        self.use_videoldm = use_videoldm 
        
        if self.use_videoldm == True:
            
            self.norm4 = nn.LayerNorm(query_dim)
            self.norm5 = nn.LayerNorm(query_dim)

            self.conv3d = Conv3DLayer(
                in_dim=query_dim,
                out_dim=query_dim,
                n_frames=self.n_frames,
            )
            #"""
            self.tempo_attn = TemporalAttentionLayer(
                    dim=query_dim,
                    n_frames=self.n_frames,
                    n_heads=n_heads,
                    kv_dim=key_dim,
                )
            #"""

        print(f"########### enable_fuser: {self.enable_fuser}, use_videoldm: {self.use_videoldm} #################")


    def forward(self, x, context, objs):
#        return checkpoint(self._forward, (x, context, objs), self.parameters(), self.use_checkpoint)
        #print("x.shape", x.shape) # torch.Size([2, 4096, 320])
        #print("context.shape", context.shape) # torch.Size([2, 77, 768])
        #print("objs.shape", objs.shape) # torch.Size([2, 30, 768]) 
        
        if self.use_checkpoint and x.requires_grad:
            #print('use checkpoint')
            return checkpoint.checkpoint(self._forward, x, context, objs)
        else:
            return self._forward(x, context, objs)

    def _forward(self, x, context, objs): 
        #print("x0.shape", x.shape) # torch.Size([2, 4096, 320])
        x = self.attn1( self.norm1(x) ) + x 
        #print("after self.attn1 x.shape", x.shape) # torch.Size([2, 4096, 320])
        if self.enable_fuser:
            x = self.fuser(x, objs) # identity mapping in the beginning 
            #print("after fuser x.shape", x.shape) # torch.Size([2, 4096, 320])
        x = self.attn2(self.norm2(x), context, context) + x
        #print("after self.attn2 x.shape", x.shape) # torch.Size([2, 4096, 320])
        x = self.ff(self.norm3(x)) + x
        #print("after ff x.shape", x.shape) # torch.Size([2, 4096, 320])
        
        if self.use_videoldm:
            hw = x.shape[1] 
            h, w = int(np.sqrt(hw)), int(np.sqrt(hw))
            
            x = self.norm4(x)
            x = rearrange(x, 'b (h w) c -> b c h w', h=h, w=w)
            
            
            x = self.conv3d(x, n_frames=self.n_frames)
            #print("after conv3d x.shape", x.shape)  # torch.Size([8, 320, 32, 32])
            
            x = rearrange(x, 'b c h w -> b h w c')
            x = self.norm5(x)
            x = rearrange(x, 'b h w c -> b c h w')
            
            x = self.tempo_attn(x, context, n_frames=self.n_frames) #+ x
            #print("after tempo attn x.shape", x.shape) # torch.Size([8, 320, 32, 32])
            
            x = rearrange(x, 'b c h w -> b (h w) c')
            
            #print("rearranged after tempo attn x.shape", x.shape) # torch.Size([8, 1024, 320])
        
        return x


class SpatialTransformer(nn.Module):
    def __init__(self, in_channels, key_dim, value_dim, n_heads, d_head, depth=1, fuser_type=None, n_frames=8, use_checkpoint=True, use_videoldm=True):
        super().__init__()
        self.in_channels = in_channels
        query_dim = n_heads * d_head
        self.norm = Normalize(in_channels)

        
        self.proj_in = nn.Conv2d(in_channels,
                                 query_dim,
                                 kernel_size=1,
                                 stride=1,
                                 padding=0)

        self.transformer_blocks = nn.ModuleList(
            [BasicTransformerBlock(query_dim, key_dim, value_dim, n_heads, d_head, fuser_type, n_frames=n_frames, 
                                   use_checkpoint=use_checkpoint, use_videoldm = use_videoldm)
                for d in range(depth)]
        )

        self.proj_out = zero_module(nn.Conv2d(query_dim,
                                              in_channels,
                                              kernel_size=1,
                                              stride=1,
                                              padding=0))

    def forward(self, x, context, objs):
        b, c, h, w = x.shape
        x_in = x
        x = self.norm(x)
        x = self.proj_in(x)
        x = rearrange(x, 'b c h w -> b (h w) c')
        for block in self.transformer_blocks:
            x = block(x, context, objs)
        x = rearrange(x, 'b (h w) c -> b c h w', h=h, w=w)
        x = self.proj_out(x)
        return x + x_in