import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.utils.data import DataLoader
from torch.cuda.amp import autocast

# Assuming necessary imports and configurations are handled externally

try:
    from spikingjelly.activation_based import functional
    SPIKINGJELLY_FUNCTIONAL_AVAILABLE = True
except ImportError:
    try:
        from spikingjelly.clock_driven import functional
        SPIKINGJELLY_FUNCTIONAL_AVAILABLE = True
    except ImportError:
        SPIKINGJELLY_FUNCTIONAL_AVAILABLE = False

def test_one_epoch(model, criterion, data_loader, device, epoch):
    """
    Perform one testing epoch over the dataset.
    
    :param model: The neural network model
    :param criterion: Loss function
    :param data_loader: Testing data loader
    :param device: Computing device
    :param epoch: Current epoch number
    :return: Average loss and accuracy
    """
    model.eval()
    total_loss = 0.0
    total_correct = 0
    total_samples = 0
    
    with torch.no_grad():
        for inputs, targets in data_loader:
            inputs = inputs.to(device).float()
            targets = targets.to(device)
            
            with autocast(enabled=True):
                logits, spike_info = model(inputs)
            
            with autocast(enabled=False):
                loss = criterion(logits, targets, spike_info=spike_info)
            
            total_loss += loss.item() * inputs.size(0)
            preds = logits.argmax(dim=1)
            total_correct += (preds == targets).sum().item()
            total_samples += inputs.size(0)
            
            if SPIKINGJELLY_FUNCTIONAL_AVAILABLE:
                functional.reset_net(model)
    
    avg_loss = total_loss / total_samples
    accuracy = total_correct / total_samples * 100
    return avg_loss, accuracy
