# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.
#
# This source code is licensed under the license found in the
# LICENSE file in the root directory of this source tree.
#
# See also denoised_mdp/envs/dmc/dmc2gym/README.md for license-related
# information about these files adapted from
# https://github.com/facebookresearch/deep_bisim4control/

from typing import *

import glob
import os
import contextlib

import cv2
from gym import core, spaces
from dm_env import specs
import numpy as np
import torch

from . import local_dm_control_suite as suite
from . import natural_imgsource
from ...abc import EnvBase, AutoResetEnvBase, IndexableSized
from .... import utils


def _spec_to_box(spec, repeat=1):
    def extract_min_max(s):
        assert s.dtype == np.float64 or s.dtype == np.float32
        dim = int(np.prod(s.shape))
        if type(s) == specs.Array:
            bound = np.inf * np.ones(dim, dtype=np.float32)
            return -bound, bound
        elif type(s) == specs.BoundedArray:
            zeros = np.zeros(dim, dtype=np.float32)
            return s.minimum + zeros, s.maximum + zeros

    mins, maxs = [], []
    for s in spec:
        mn, mx = extract_min_max(s)
        mins.append(mn)
        maxs.append(mx)
    low = np.tile(np.concatenate(mins, axis=0), repeat).astype(np.float32)
    high = np.tile(np.concatenate(maxs, axis=0), repeat).astype(np.float32)
    assert low.shape == high.shape
    return spaces.Box(low, high, dtype=np.float32)


def _flatten_obs(obs, dtype=None):
    obs_pieces = []
    for v in obs.values():
        if np.isscalar(v):
            flat = np.array([v], dtype=dtype)
        else:
            flat = np.asarray(v.ravel(), dtype=dtype)
        obs_pieces.append(flat)
    return np.concatenate(obs_pieces, axis=0)


class DMCWrapper(core.Env, EnvBase):

    @property
    def max_episode_length(self) -> int:
        return self._episode_length

    @property
    def observation_output_kind(self) -> EnvBase.ObsOutputKind:
        return self._observation_output_kind

    @property
    def action_repeat(self) -> int:
        return self._frame_skip

    def __init__(
        self,
        domain_name,
        task_name,
        total_frames,
        task_kwargs={},
        visualize_reward={},
        height=84,
        width=84,
        camera_id=0,
        frame_skip=1,
        environment_kwargs=None,
        observation_output_kind=EnvBase.ObsOutputKind.image_uint8,
        seed=1,
        episode_length=1000,
    ):

        self._height = height
        self._width = width
        self._camera_id = camera_id
        self._frame_skip = frame_skip
        self._observation_output_kind = observation_output_kind
        self._episode_length = episode_length

        # create task
        self._env = suite.load(
            domain_name=domain_name,
            task_name=task_name,
            task_kwargs=task_kwargs,
            visualize_reward=visualize_reward,
            environment_kwargs=environment_kwargs,
        )        

        # true and normalized action spaces
        self._true_action_space = _spec_to_box([self._env.action_spec()])
        self._norm_action_space = spaces.Box(
            low=-1.0,
            high=1.0,
            shape=self._true_action_space.shape,
            dtype=np.float32,
        )

        # create observation space
        self._observation_space = EnvBase.ObsOutputKind.get_observation_space(observation_output_kind, height, width)

        self._internal_state_space = spaces.Box(
            low=-np.inf,
            high=np.inf,
            shape=tuple(self._env.physics.get_state().shape),
            dtype=np.float32,
        )

        self._steps_taken = 0
        self.seed(seed)

    def __getattr__(self, name):
        return getattr(self._env, name)

    def _get_obs(self):
        obs = self.render(
            height=self._height,
            width=self._width,
            camera_id=self._camera_id,
        )
        obs = self.ndarray_uint8_image_to_observation(obs, target_shape=None)

        return obs

    def _convert_action(self, action):
        action = action.astype(np.float64)
        true_delta = self._true_action_space.high - self._true_action_space.low
        norm_delta = self._norm_action_space.high - self._norm_action_space.low
        action = (action - self._norm_action_space.low) / norm_delta
        action = action * true_delta + self._true_action_space.low
        action = action.astype(np.float32)
        return action

    @property
    def observation_space(self):
        return self._observation_space

    @property
    def internal_state_space(self):
        return self._internal_state_space

    @property
    def action_space(self):
        return self._norm_action_space

    def sample_random_action(self, size=()):
        return torch.as_tensor(np.random.uniform(-1, 1, size=tuple(size) + tuple(self.action_shape)), dtype=torch.float32)

    @property
    def dmc_env_np_random_state(self) -> np.random.RandomState:
        return cast(np.random.RandomState, self._env.task.random)

    def seed(self, seed: int):
        self.dmc_env_np_random_state.seed(seed)

    def randn(self, *size: int) -> torch.Tensor:
        return torch.as_tensor(self.randn_np(*size))

    def randn_np(self, *size: int) -> np.ndarray:
        return np.random.normal(size=size)

    def get_random_state(self):
        return self.dmc_env_np_random_state.get_state()

    def set_random_state(self, random_state):
        self.dmc_env_np_random_state.set_state(random_state)

    def _step_env(self, env, action):
        ts = env.step(action)
        return ts

    def step(self, action):
        if isinstance(action, torch.Tensor):
            action = action.detach()
        action = np.asarray(action, dtype=np.float32)
        assert self._norm_action_space.contains(action)
        action = self._convert_action(action)
        assert self._true_action_space.contains(action)
        reward = 0
        last_reward = 0
        actual_env_steps_taken = 0

        for _ in range(self._frame_skip):
            timestep = self._step_env(self._env, action)
            last_reward = timestep.reward or 0
            reward += last_reward
            done = timestep.last()
            actual_env_steps_taken += 1
            self._steps_taken += 1
            if self._steps_taken >= self._episode_length:
                done = True
            if done:
                break
        obs = self._get_obs()
        return obs, reward, done, EnvBase.Info(actual_env_steps_taken)  #, extra

    def reset(self) -> Tuple[torch.Tensor, EnvBase.Info]:
        self._steps_taken = 0
        timestep = self._env.reset()
        obs = self._get_obs()
        return obs, EnvBase.Info(0)

    def render(self, mode='rgb_array', height=None, width=None, camera_id=0):
        assert mode == 'rgb_array', 'only support rgb_array mode, given %s' % mode
        height = height or self._height
        width = width or self._width
        camera_id = camera_id or self._camera_id
        return self._env.physics.render(height=height, width=width, camera_id=camera_id)
