# Treat a cube as a bitmap, and create the index into this array in order of
# ZYX (note Z is the most significant digit).
# The resulting object is an array of triangles, where each triangle is 6
# indices. Each consecutive pair of indices within this triangle represents an
# edge spanning two corners (identified by the indices).
#
# The corners of a cube are indexed as follows
#
#    (0, 0, 0), (1, 0, 0), (0, 1, 0), (1, 1, 0),
#    (0, 0, 1), (1, 0, 1), (0, 1, 1), (1, 1, 1)
#
# Here is a visualization of the cube indices:
#
#        6 + -----------------------+ 7
#         /|                       /|
#        / |                      / |
#       /  |                     /  |
#    4 +------------------------+ 5 |
#      |   |                    |   |
#      |   |                    |   |
#      |   |                    |   |
#      |   | 2                  |   | 3
#      |   +--------------------|---+
#      |  /                     |  /
#      | /                      | /
#      |/                       |/
#      +------------------------+
#     0                           1
#
# Derived using model3d, in particular this function:
# https://github.com/unixpickle/model3d/blob/7a3adb982c154c80c1a22032b5a0695160a7f96d/model3d/mc.go#L434
#
MC_TABLE = [
    [],
    [[0, 1, 0, 2, 0, 4]],
    [[1, 0, 1, 5, 1, 3]],
    [[0, 4, 1, 5, 0, 2], [1, 5, 1, 3, 0, 2]],
    [[2, 0, 2, 3, 2, 6]],
    [[0, 1, 2, 3, 0, 4], [2, 3, 2, 6, 0, 4]],
    [[1, 0, 1, 5, 1, 3], [2, 6, 0, 2, 3, 2]],
    [[3, 2, 2, 6, 3, 1], [3, 1, 2, 6, 1, 5], [1, 5, 2, 6, 0, 4]],
    [[3, 1, 3, 7, 3, 2]],
    [[0, 2, 0, 4, 0, 1], [3, 7, 2, 3, 1, 3]],
    [[1, 5, 3, 7, 1, 0], [3, 7, 3, 2, 1, 0]],
    [[2, 0, 0, 4, 2, 3], [2, 3, 0, 4, 3, 7], [3, 7, 0, 4, 1, 5]],
    [[2, 0, 3, 1, 2, 6], [3, 1, 3, 7, 2, 6]],
    [[1, 3, 3, 7, 1, 0], [1, 0, 3, 7, 0, 4], [0, 4, 3, 7, 2, 6]],
    [[0, 1, 1, 5, 0, 2], [0, 2, 1, 5, 2, 6], [2, 6, 1, 5, 3, 7]],
    [[0, 4, 1, 5, 3, 7], [0, 4, 3, 7, 2, 6]],
    [[4, 0, 4, 6, 4, 5]],
    [[0, 2, 4, 6, 0, 1], [4, 6, 4, 5, 0, 1]],
    [[1, 5, 1, 3, 1, 0], [4, 6, 5, 4, 0, 4]],
    [[5, 1, 1, 3, 5, 4], [5, 4, 1, 3, 4, 6], [4, 6, 1, 3, 0, 2]],
    [[2, 0, 2, 3, 2, 6], [4, 5, 0, 4, 6, 4]],
    [[6, 4, 4, 5, 6, 2], [6, 2, 4, 5, 2, 3], [2, 3, 4, 5, 0, 1]],
    [[2, 6, 2, 0, 3, 2], [1, 0, 1, 5, 3, 1], [6, 4, 5, 4, 0, 4]],
    [[1, 3, 5, 4, 1, 5], [1, 3, 4, 6, 5, 4], [1, 3, 3, 2, 4, 6], [3, 2, 2, 6, 4, 6]],
    [[3, 1, 3, 7, 3, 2], [6, 4, 5, 4, 0, 4]],
    [[4, 5, 0, 1, 4, 6], [0, 1, 0, 2, 4, 6], [7, 3, 2, 3, 1, 3]],
    [[3, 2, 1, 0, 3, 7], [1, 0, 1, 5, 3, 7], [6, 4, 5, 4, 0, 4]],
    [[3, 7, 3, 2, 1, 5], [3, 2, 6, 4, 1, 5], [1, 5, 6, 4, 5, 4], [3, 2, 2, 0, 6, 4]],
    [[3, 7, 2, 6, 3, 1], [2, 6, 2, 0, 3, 1], [5, 4, 0, 4, 6, 4]],
    [[1, 0, 1, 3, 5, 4], [1, 3, 2, 6, 5, 4], [1, 3, 3, 7, 2, 6], [5, 4, 2, 6, 4, 6]],
    [[0, 1, 1, 5, 0, 2], [0, 2, 1, 5, 2, 6], [2, 6, 1, 5, 3, 7], [4, 5, 0, 4, 4, 6]],
    [[6, 2, 4, 6, 4, 5], [4, 5, 5, 1, 6, 2], [6, 2, 5, 1, 7, 3]],
    [[5, 1, 5, 4, 5, 7]],
    [[0, 1, 0, 2, 0, 4], [5, 7, 1, 5, 4, 5]],
    [[1, 0, 5, 4, 1, 3], [5, 4, 5, 7, 1, 3]],
    [[4, 5, 5, 7, 4, 0], [4, 0, 5, 7, 0, 2], [0, 2, 5, 7, 1, 3]],
    [[2, 0, 2, 3, 2, 6], [7, 5, 1, 5, 4, 5]],
    [[2, 6, 0, 4, 2, 3], [0, 4, 0, 1, 2, 3], [7, 5, 1, 5, 4, 5]],
    [[5, 7, 1, 3, 5, 4], [1, 3, 1, 0, 5, 4], [6, 2, 0, 2, 3, 2]],
    [[3, 1, 3, 2, 7, 5], [3, 2, 0, 4, 7, 5], [3, 2, 2, 6, 0, 4], [7, 5, 0, 4, 5, 4]],
    [[3, 7, 3, 2, 3, 1], [5, 4, 7, 5, 1, 5]],
    [[0, 4, 0, 1, 2, 0], [3, 1, 3, 7, 2, 3], [4, 5, 7, 5, 1, 5]],
    [[7, 3, 3, 2, 7, 5], [7, 5, 3, 2, 5, 4], [5, 4, 3, 2, 1, 0]],
    [[0, 4, 2, 3, 0, 2], [0, 4, 3, 7, 2, 3], [0, 4, 4, 5, 3, 7], [4, 5, 5, 7, 3, 7]],
    [[2, 0, 3, 1, 2, 6], [3, 1, 3, 7, 2, 6], [4, 5, 7, 5, 1, 5]],
    [[1, 3, 3, 7, 1, 0], [1, 0, 3, 7, 0, 4], [0, 4, 3, 7, 2, 6], [5, 7, 1, 5, 5, 4]],
    [[2, 6, 2, 0, 3, 7], [2, 0, 4, 5, 3, 7], [3, 7, 4, 5, 7, 5], [2, 0, 0, 1, 4, 5]],
    [[4, 0, 5, 4, 5, 7], [5, 7, 7, 3, 4, 0], [4, 0, 7, 3, 6, 2]],
    [[4, 6, 5, 7, 4, 0], [5, 7, 5, 1, 4, 0]],
    [[1, 0, 0, 2, 1, 5], [1, 5, 0, 2, 5, 7], [5, 7, 0, 2, 4, 6]],
    [[0, 4, 4, 6, 0, 1], [0, 1, 4, 6, 1, 3], [1, 3, 4, 6, 5, 7]],
    [[0, 2, 4, 6, 5, 7], [0, 2, 5, 7, 1, 3]],
    [[5, 1, 4, 0, 5, 7], [4, 0, 4, 6, 5, 7], [3, 2, 6, 2, 0, 2]],
    [[2, 3, 2, 6, 0, 1], [2, 6, 7, 5, 0, 1], [0, 1, 7, 5, 1, 5], [2, 6, 6, 4, 7, 5]],
    [[0, 4, 4, 6, 0, 1], [0, 1, 4, 6, 1, 3], [1, 3, 4, 6, 5, 7], [2, 6, 0, 2, 2, 3]],
    [[3, 1, 2, 3, 2, 6], [2, 6, 6, 4, 3, 1], [3, 1, 6, 4, 7, 5]],
    [[4, 6, 5, 7, 4, 0], [5, 7, 5, 1, 4, 0], [2, 3, 1, 3, 7, 3]],
    [[1, 0, 0, 2, 1, 5], [1, 5, 0, 2, 5, 7], [5, 7, 0, 2, 4, 6], [3, 2, 1, 3, 3, 7]],
    [[0, 1, 0, 4, 2, 3], [0, 4, 5, 7, 2, 3], [0, 4, 4, 6, 5, 7], [2, 3, 5, 7, 3, 7]],
    [[7, 5, 3, 7, 3, 2], [3, 2, 2, 0, 7, 5], [7, 5, 2, 0, 6, 4]],
    [[0, 4, 4, 6, 5, 7], [0, 4, 5, 7, 1, 5], [0, 2, 1, 3, 3, 7], [3, 7, 2, 6, 0, 2]],
    [
        [3, 1, 7, 3, 6, 2],
        [6, 2, 0, 1, 3, 1],
        [6, 4, 0, 1, 6, 2],
        [6, 4, 5, 1, 0, 1],
        [6, 4, 7, 5, 5, 1],
    ],
    [
        [4, 0, 6, 4, 7, 5],
        [7, 5, 1, 0, 4, 0],
        [7, 3, 1, 0, 7, 5],
        [7, 3, 2, 0, 1, 0],
        [7, 3, 6, 2, 2, 0],
    ],
    [[7, 3, 6, 2, 6, 4], [7, 5, 7, 3, 6, 4]],
    [[6, 2, 6, 7, 6, 4]],
    [[0, 4, 0, 1, 0, 2], [6, 7, 4, 6, 2, 6]],
    [[1, 0, 1, 5, 1, 3], [7, 6, 4, 6, 2, 6]],
    [[1, 3, 0, 2, 1, 5], [0, 2, 0, 4, 1, 5], [7, 6, 4, 6, 2, 6]],
    [[2, 3, 6, 7, 2, 0], [6, 7, 6, 4, 2, 0]],
    [[4, 0, 0, 1, 4, 6], [4, 6, 0, 1, 6, 7], [6, 7, 0, 1, 2, 3]],
    [[6, 4, 2, 0, 6, 7], [2, 0, 2, 3, 6, 7], [5, 1, 3, 1, 0, 1]],
    [[1, 5, 1, 3, 0, 4], [1, 3, 7, 6, 0, 4], [0, 4, 7, 6, 4, 6], [1, 3, 3, 2, 7, 6]],
    [[3, 2, 3, 1, 3, 7], [6, 4, 2, 6, 7, 6]],
    [[3, 7, 3, 2, 1, 3], [0, 2, 0, 4, 1, 0], [7, 6, 4, 6, 2, 6]],
    [[1, 5, 3, 7, 1, 0], [3, 7, 3, 2, 1, 0], [4, 6, 2, 6, 7, 6]],
    [[2, 0, 0, 4, 2, 3], [2, 3, 0, 4, 3, 7], [3, 7, 0, 4, 1, 5], [6, 4, 2, 6, 6, 7]],
    [[7, 6, 6, 4, 7, 3], [7, 3, 6, 4, 3, 1], [3, 1, 6, 4, 2, 0]],
    [[0, 1, 4, 6, 0, 4], [0, 1, 6, 7, 4, 6], [0, 1, 1, 3, 6, 7], [1, 3, 3, 7, 6, 7]],
    [[0, 2, 0, 1, 4, 6], [0, 1, 3, 7, 4, 6], [0, 1, 1, 5, 3, 7], [4, 6, 3, 7, 6, 7]],
    [[7, 3, 6, 7, 6, 4], [6, 4, 4, 0, 7, 3], [7, 3, 4, 0, 5, 1]],
    [[4, 0, 6, 2, 4, 5], [6, 2, 6, 7, 4, 5]],
    [[2, 6, 6, 7, 2, 0], [2, 0, 6, 7, 0, 1], [0, 1, 6, 7, 4, 5]],
    [[6, 7, 4, 5, 6, 2], [4, 5, 4, 0, 6, 2], [3, 1, 0, 1, 5, 1]],
    [[2, 0, 2, 6, 3, 1], [2, 6, 4, 5, 3, 1], [2, 6, 6, 7, 4, 5], [3, 1, 4, 5, 1, 5]],
    [[0, 2, 2, 3, 0, 4], [0, 4, 2, 3, 4, 5], [4, 5, 2, 3, 6, 7]],
    [[0, 1, 2, 3, 6, 7], [0, 1, 6, 7, 4, 5]],
    [[0, 2, 2, 3, 0, 4], [0, 4, 2, 3, 4, 5], [4, 5, 2, 3, 6, 7], [1, 3, 0, 1, 1, 5]],
    [[5, 4, 1, 5, 1, 3], [1, 3, 3, 2, 5, 4], [5, 4, 3, 2, 7, 6]],
    [[4, 0, 6, 2, 4, 5], [6, 2, 6, 7, 4, 5], [1, 3, 7, 3, 2, 3]],
    [[2, 6, 6, 7, 2, 0], [2, 0, 6, 7, 0, 1], [0, 1, 6, 7, 4, 5], [3, 7, 2, 3, 3, 1]],
    [[0, 1, 1, 5, 3, 7], [0, 1, 3, 7, 2, 3], [0, 4, 2, 6, 6, 7], [6, 7, 4, 5, 0, 4]],
    [
        [6, 2, 7, 6, 5, 4],
        [5, 4, 0, 2, 6, 2],
        [5, 1, 0, 2, 5, 4],
        [5, 1, 3, 2, 0, 2],
        [5, 1, 7, 3, 3, 2],
    ],
    [[3, 1, 3, 7, 2, 0], [3, 7, 5, 4, 2, 0], [2, 0, 5, 4, 0, 4], [3, 7, 7, 6, 5, 4]],
    [[1, 0, 3, 1, 3, 7], [3, 7, 7, 6, 1, 0], [1, 0, 7, 6, 5, 4]],
    [
        [1, 0, 5, 1, 7, 3],
        [7, 3, 2, 0, 1, 0],
        [7, 6, 2, 0, 7, 3],
        [7, 6, 4, 0, 2, 0],
        [7, 6, 5, 4, 4, 0],
    ],
    [[7, 6, 5, 4, 5, 1], [7, 3, 7, 6, 5, 1]],
    [[5, 7, 5, 1, 5, 4], [6, 2, 7, 6, 4, 6]],
    [[0, 2, 0, 4, 1, 0], [5, 4, 5, 7, 1, 5], [2, 6, 7, 6, 4, 6]],
    [[1, 0, 5, 4, 1, 3], [5, 4, 5, 7, 1, 3], [2, 6, 7, 6, 4, 6]],
    [[4, 5, 5, 7, 4, 0], [4, 0, 5, 7, 0, 2], [0, 2, 5, 7, 1, 3], [6, 7, 4, 6, 6, 2]],
    [[2, 3, 6, 7, 2, 0], [6, 7, 6, 4, 2, 0], [1, 5, 4, 5, 7, 5]],
    [[4, 0, 0, 1, 4, 6], [4, 6, 0, 1, 6, 7], [6, 7, 0, 1, 2, 3], [5, 1, 4, 5, 5, 7]],
    [[0, 2, 2, 3, 6, 7], [0, 2, 6, 7, 4, 6], [0, 1, 4, 5, 5, 7], [5, 7, 1, 3, 0, 1]],
    [
        [5, 4, 7, 5, 3, 1],
        [3, 1, 0, 4, 5, 4],
        [3, 2, 0, 4, 3, 1],
        [3, 2, 6, 4, 0, 4],
        [3, 2, 7, 6, 6, 4],
    ],
    [[5, 4, 5, 7, 1, 5], [3, 7, 3, 2, 1, 3], [4, 6, 2, 6, 7, 6]],
    [[1, 0, 0, 2, 0, 4], [1, 5, 5, 4, 5, 7], [3, 2, 1, 3, 3, 7], [2, 6, 7, 6, 4, 6]],
    [[7, 3, 3, 2, 7, 5], [7, 5, 3, 2, 5, 4], [5, 4, 3, 2, 1, 0], [6, 2, 7, 6, 6, 4]],
    [
        [0, 4, 2, 3, 0, 2],
        [0, 4, 3, 7, 2, 3],
        [0, 4, 4, 5, 3, 7],
        [4, 5, 5, 7, 3, 7],
        [6, 7, 4, 6, 2, 6],
    ],
    [[7, 6, 6, 4, 7, 3], [7, 3, 6, 4, 3, 1], [3, 1, 6, 4, 2, 0], [5, 4, 7, 5, 5, 1]],
    [
        [0, 1, 4, 6, 0, 4],
        [0, 1, 6, 7, 4, 6],
        [0, 1, 1, 3, 6, 7],
        [1, 3, 3, 7, 6, 7],
        [5, 7, 1, 5, 4, 5],
    ],
    [
        [6, 7, 4, 6, 0, 2],
        [0, 2, 3, 7, 6, 7],
        [0, 1, 3, 7, 0, 2],
        [0, 1, 5, 7, 3, 7],
        [0, 1, 4, 5, 5, 7],
    ],
    [[4, 0, 6, 7, 4, 6], [4, 0, 7, 3, 6, 7], [4, 0, 5, 7, 7, 3], [4, 5, 5, 7, 4, 0]],
    [[7, 5, 5, 1, 7, 6], [7, 6, 5, 1, 6, 2], [6, 2, 5, 1, 4, 0]],
    [[0, 2, 1, 5, 0, 1], [0, 2, 5, 7, 1, 5], [0, 2, 2, 6, 5, 7], [2, 6, 6, 7, 5, 7]],
    [[1, 3, 1, 0, 5, 7], [1, 0, 2, 6, 5, 7], [5, 7, 2, 6, 7, 6], [1, 0, 0, 4, 2, 6]],
    [[2, 0, 6, 2, 6, 7], [6, 7, 7, 5, 2, 0], [2, 0, 7, 5, 3, 1]],
    [[0, 4, 0, 2, 1, 5], [0, 2, 6, 7, 1, 5], [0, 2, 2, 3, 6, 7], [1, 5, 6, 7, 5, 7]],
    [[7, 6, 5, 7, 5, 1], [5, 1, 1, 0, 7, 6], [7, 6, 1, 0, 3, 2]],
    [
        [2, 0, 3, 2, 7, 6],
        [7, 6, 4, 0, 2, 0],
        [7, 5, 4, 0, 7, 6],
        [7, 5, 1, 0, 4, 0],
        [7, 5, 3, 1, 1, 0],
    ],
    [[7, 5, 3, 1, 3, 2], [7, 6, 7, 5, 3, 2]],
    [[7, 5, 5, 1, 7, 6], [7, 6, 5, 1, 6, 2], [6, 2, 5, 1, 4, 0], [3, 1, 7, 3, 3, 2]],
    [
        [0, 2, 1, 5, 0, 1],
        [0, 2, 5, 7, 1, 5],
        [0, 2, 2, 6, 5, 7],
        [2, 6, 6, 7, 5, 7],
        [3, 7, 2, 3, 1, 3],
    ],
    [
        [3, 7, 2, 3, 0, 1],
        [0, 1, 5, 7, 3, 7],
        [0, 4, 5, 7, 0, 1],
        [0, 4, 6, 7, 5, 7],
        [0, 4, 2, 6, 6, 7],
    ],
    [[2, 0, 3, 7, 2, 3], [2, 0, 7, 5, 3, 7], [2, 0, 6, 7, 7, 5], [2, 6, 6, 7, 2, 0]],
    [
        [5, 7, 1, 5, 0, 4],
        [0, 4, 6, 7, 5, 7],
        [0, 2, 6, 7, 0, 4],
        [0, 2, 3, 7, 6, 7],
        [0, 2, 1, 3, 3, 7],
    ],
    [[1, 0, 5, 7, 1, 5], [1, 0, 7, 6, 5, 7], [1, 0, 3, 7, 7, 6], [1, 3, 3, 7, 1, 0]],
    [[0, 2, 0, 1, 0, 4], [3, 7, 6, 7, 5, 7]],
    [[7, 5, 7, 3, 7, 6]],
    [[7, 3, 7, 5, 7, 6]],
    [[0, 1, 0, 2, 0, 4], [6, 7, 3, 7, 5, 7]],
    [[1, 3, 1, 0, 1, 5], [7, 6, 3, 7, 5, 7]],
    [[0, 4, 1, 5, 0, 2], [1, 5, 1, 3, 0, 2], [6, 7, 3, 7, 5, 7]],
    [[2, 6, 2, 0, 2, 3], [7, 5, 6, 7, 3, 7]],
    [[0, 1, 2, 3, 0, 4], [2, 3, 2, 6, 0, 4], [5, 7, 6, 7, 3, 7]],
    [[1, 5, 1, 3, 0, 1], [2, 3, 2, 6, 0, 2], [5, 7, 6, 7, 3, 7]],
    [[3, 2, 2, 6, 3, 1], [3, 1, 2, 6, 1, 5], [1, 5, 2, 6, 0, 4], [7, 6, 3, 7, 7, 5]],
    [[3, 1, 7, 5, 3, 2], [7, 5, 7, 6, 3, 2]],
    [[7, 6, 3, 2, 7, 5], [3, 2, 3, 1, 7, 5], [4, 0, 1, 0, 2, 0]],
    [[5, 7, 7, 6, 5, 1], [5, 1, 7, 6, 1, 0], [1, 0, 7, 6, 3, 2]],
    [[2, 3, 2, 0, 6, 7], [2, 0, 1, 5, 6, 7], [2, 0, 0, 4, 1, 5], [6, 7, 1, 5, 7, 5]],
    [[6, 2, 2, 0, 6, 7], [6, 7, 2, 0, 7, 5], [7, 5, 2, 0, 3, 1]],
    [[0, 4, 0, 1, 2, 6], [0, 1, 5, 7, 2, 6], [2, 6, 5, 7, 6, 7], [0, 1, 1, 3, 5, 7]],
    [[1, 5, 0, 2, 1, 0], [1, 5, 2, 6, 0, 2], [1, 5, 5, 7, 2, 6], [5, 7, 7, 6, 2, 6]],
    [[5, 1, 7, 5, 7, 6], [7, 6, 6, 2, 5, 1], [5, 1, 6, 2, 4, 0]],
    [[4, 5, 4, 0, 4, 6], [7, 3, 5, 7, 6, 7]],
    [[0, 2, 4, 6, 0, 1], [4, 6, 4, 5, 0, 1], [3, 7, 5, 7, 6, 7]],
    [[4, 6, 4, 5, 0, 4], [1, 5, 1, 3, 0, 1], [6, 7, 3, 7, 5, 7]],
    [[5, 1, 1, 3, 5, 4], [5, 4, 1, 3, 4, 6], [4, 6, 1, 3, 0, 2], [7, 3, 5, 7, 7, 6]],
    [[2, 3, 2, 6, 0, 2], [4, 6, 4, 5, 0, 4], [3, 7, 5, 7, 6, 7]],
    [[6, 4, 4, 5, 6, 2], [6, 2, 4, 5, 2, 3], [2, 3, 4, 5, 0, 1], [7, 5, 6, 7, 7, 3]],
    [[0, 1, 1, 5, 1, 3], [0, 2, 2, 3, 2, 6], [4, 5, 0, 4, 4, 6], [5, 7, 6, 7, 3, 7]],
    [
        [1, 3, 5, 4, 1, 5],
        [1, 3, 4, 6, 5, 4],
        [1, 3, 3, 2, 4, 6],
        [3, 2, 2, 6, 4, 6],
        [7, 6, 3, 7, 5, 7],
    ],
    [[3, 1, 7, 5, 3, 2], [7, 5, 7, 6, 3, 2], [0, 4, 6, 4, 5, 4]],
    [[1, 0, 0, 2, 4, 6], [1, 0, 4, 6, 5, 4], [1, 3, 5, 7, 7, 6], [7, 6, 3, 2, 1, 3]],
    [[5, 7, 7, 6, 5, 1], [5, 1, 7, 6, 1, 0], [1, 0, 7, 6, 3, 2], [4, 6, 5, 4, 4, 0]],
    [
        [7, 5, 6, 7, 2, 3],
        [2, 3, 1, 5, 7, 5],
        [2, 0, 1, 5, 2, 3],
        [2, 0, 4, 5, 1, 5],
        [2, 0, 6, 4, 4, 5],
    ],
    [[6, 2, 2, 0, 6, 7], [6, 7, 2, 0, 7, 5], [7, 5, 2, 0, 3, 1], [4, 0, 6, 4, 4, 5]],
    [
        [4, 6, 5, 4, 1, 0],
        [1, 0, 2, 6, 4, 6],
        [1, 3, 2, 6, 1, 0],
        [1, 3, 7, 6, 2, 6],
        [1, 3, 5, 7, 7, 6],
    ],
    [
        [1, 5, 0, 2, 1, 0],
        [1, 5, 2, 6, 0, 2],
        [1, 5, 5, 7, 2, 6],
        [5, 7, 7, 6, 2, 6],
        [4, 6, 5, 4, 0, 4],
    ],
    [[5, 1, 4, 6, 5, 4], [5, 1, 6, 2, 4, 6], [5, 1, 7, 6, 6, 2], [5, 7, 7, 6, 5, 1]],
    [[5, 4, 7, 6, 5, 1], [7, 6, 7, 3, 5, 1]],
    [[7, 3, 5, 1, 7, 6], [5, 1, 5, 4, 7, 6], [2, 0, 4, 0, 1, 0]],
    [[3, 1, 1, 0, 3, 7], [3, 7, 1, 0, 7, 6], [7, 6, 1, 0, 5, 4]],
    [[0, 2, 0, 4, 1, 3], [0, 4, 6, 7, 1, 3], [1, 3, 6, 7, 3, 7], [0, 4, 4, 5, 6, 7]],
    [[5, 4, 7, 6, 5, 1], [7, 6, 7, 3, 5, 1], [0, 2, 3, 2, 6, 2]],
    [[1, 5, 5, 4, 7, 6], [1, 5, 7, 6, 3, 7], [1, 0, 3, 2, 2, 6], [2, 6, 0, 4, 1, 0]],
    [[3, 1, 1, 0, 3, 7], [3, 7, 1, 0, 7, 6], [7, 6, 1, 0, 5, 4], [2, 0, 3, 2, 2, 6]],
    [
        [2, 3, 6, 2, 4, 0],
        [4, 0, 1, 3, 2, 3],
        [4, 5, 1, 3, 4, 0],
        [4, 5, 7, 3, 1, 3],
        [4, 5, 6, 7, 7, 3],
    ],
    [[1, 5, 5, 4, 1, 3], [1, 3, 5, 4, 3, 2], [3, 2, 5, 4, 7, 6]],
    [[1, 5, 5, 4, 1, 3], [1, 3, 5, 4, 3, 2], [3, 2, 5, 4, 7, 6], [0, 4, 1, 0, 0, 2]],
    [[1, 0, 5, 4, 7, 6], [1, 0, 7, 6, 3, 2]],
    [[2, 3, 0, 2, 0, 4], [0, 4, 4, 5, 2, 3], [2, 3, 4, 5, 6, 7]],
    [[1, 3, 1, 5, 0, 2], [1, 5, 7, 6, 0, 2], [1, 5, 5, 4, 7, 6], [0, 2, 7, 6, 2, 6]],
    [
        [5, 1, 4, 5, 6, 7],
        [6, 7, 3, 1, 5, 1],
        [6, 2, 3, 1, 6, 7],
        [6, 2, 0, 1, 3, 1],
        [6, 2, 4, 0, 0, 1],
    ],
    [[6, 7, 2, 6, 2, 0], [2, 0, 0, 1, 6, 7], [6, 7, 0, 1, 4, 5]],
    [[6, 2, 4, 0, 4, 5], [6, 7, 6, 2, 4, 5]],
    [[6, 7, 7, 3, 6, 4], [6, 4, 7, 3, 4, 0], [4, 0, 7, 3, 5, 1]],
    [[1, 5, 1, 0, 3, 7], [1, 0, 4, 6, 3, 7], [1, 0, 0, 2, 4, 6], [3, 7, 4, 6, 7, 6]],
    [[1, 0, 3, 7, 1, 3], [1, 0, 7, 6, 3, 7], [1, 0, 0, 4, 7, 6], [0, 4, 4, 6, 7, 6]],
    [[6, 4, 7, 6, 7, 3], [7, 3, 3, 1, 6, 4], [6, 4, 3, 1, 2, 0]],
    [[6, 7, 7, 3, 6, 4], [6, 4, 7, 3, 4, 0], [4, 0, 7, 3, 5, 1], [2, 3, 6, 2, 2, 0]],
    [
        [7, 6, 3, 7, 1, 5],
        [1, 5, 4, 6, 7, 6],
        [1, 0, 4, 6, 1, 5],
        [1, 0, 2, 6, 4, 6],
        [1, 0, 3, 2, 2, 6],
    ],
    [
        [1, 0, 3, 7, 1, 3],
        [1, 0, 7, 6, 3, 7],
        [1, 0, 0, 4, 7, 6],
        [0, 4, 4, 6, 7, 6],
        [2, 6, 0, 2, 3, 2],
    ],
    [[3, 1, 7, 6, 3, 7], [3, 1, 6, 4, 7, 6], [3, 1, 2, 6, 6, 4], [3, 2, 2, 6, 3, 1]],
    [[3, 2, 3, 1, 7, 6], [3, 1, 0, 4, 7, 6], [7, 6, 0, 4, 6, 4], [3, 1, 1, 5, 0, 4]],
    [
        [0, 1, 2, 0, 6, 4],
        [6, 4, 5, 1, 0, 1],
        [6, 7, 5, 1, 6, 4],
        [6, 7, 3, 1, 5, 1],
        [6, 7, 2, 3, 3, 1],
    ],
    [[0, 1, 4, 0, 4, 6], [4, 6, 6, 7, 0, 1], [0, 1, 6, 7, 2, 3]],
    [[6, 7, 2, 3, 2, 0], [6, 4, 6, 7, 2, 0]],
    [
        [2, 6, 0, 2, 1, 3],
        [1, 3, 7, 6, 2, 6],
        [1, 5, 7, 6, 1, 3],
        [1, 5, 4, 6, 7, 6],
        [1, 5, 0, 4, 4, 6],
    ],
    [[1, 5, 1, 0, 1, 3], [4, 6, 7, 6, 2, 6]],
    [[0, 1, 2, 6, 0, 2], [0, 1, 6, 7, 2, 6], [0, 1, 4, 6, 6, 7], [0, 4, 4, 6, 0, 1]],
    [[6, 7, 6, 2, 6, 4]],
    [[6, 2, 7, 3, 6, 4], [7, 3, 7, 5, 6, 4]],
    [[7, 5, 6, 4, 7, 3], [6, 4, 6, 2, 7, 3], [1, 0, 2, 0, 4, 0]],
    [[6, 2, 7, 3, 6, 4], [7, 3, 7, 5, 6, 4], [0, 1, 5, 1, 3, 1]],
    [[2, 0, 0, 4, 1, 5], [2, 0, 1, 5, 3, 1], [2, 6, 3, 7, 7, 5], [7, 5, 6, 4, 2, 6]],
    [[3, 7, 7, 5, 3, 2], [3, 2, 7, 5, 2, 0], [2, 0, 7, 5, 6, 4]],
    [[3, 2, 3, 7, 1, 0], [3, 7, 6, 4, 1, 0], [3, 7, 7, 5, 6, 4], [1, 0, 6, 4, 0, 4]],
    [[3, 7, 7, 5, 3, 2], [3, 2, 7, 5, 2, 0], [2, 0, 7, 5, 6, 4], [1, 5, 3, 1, 1, 0]],
    [
        [7, 3, 5, 7, 4, 6],
        [4, 6, 2, 3, 7, 3],
        [4, 0, 2, 3, 4, 6],
        [4, 0, 1, 3, 2, 3],
        [4, 0, 5, 1, 1, 3],
    ],
    [[2, 3, 3, 1, 2, 6], [2, 6, 3, 1, 6, 4], [6, 4, 3, 1, 7, 5]],
    [[2, 3, 3, 1, 2, 6], [2, 6, 3, 1, 6, 4], [6, 4, 3, 1, 7, 5], [0, 1, 2, 0, 0, 4]],
    [[1, 0, 1, 5, 3, 2], [1, 5, 4, 6, 3, 2], [3, 2, 4, 6, 2, 6], [1, 5, 5, 7, 4, 6]],
    [
        [0, 2, 4, 0, 5, 1],
        [5, 1, 3, 2, 0, 2],
        [5, 7, 3, 2, 5, 1],
        [5, 7, 6, 2, 3, 2],
        [5, 7, 4, 6, 6, 2],
    ],
    [[2, 0, 3, 1, 7, 5], [2, 0, 7, 5, 6, 4]],
    [[4, 6, 0, 4, 0, 1], [0, 1, 1, 3, 4, 6], [4, 6, 1, 3, 5, 7]],
    [[0, 2, 1, 0, 1, 5], [1, 5, 5, 7, 0, 2], [0, 2, 5, 7, 4, 6]],
    [[5, 7, 4, 6, 4, 0], [5, 1, 5, 7, 4, 0]],
    [[5, 4, 4, 0, 5, 7], [5, 7, 4, 0, 7, 3], [7, 3, 4, 0, 6, 2]],
    [[0, 1, 0, 2, 4, 5], [0, 2, 3, 7, 4, 5], [4, 5, 3, 7, 5, 7], [0, 2, 2, 6, 3, 7]],
    [[5, 4, 4, 0, 5, 7], [5, 7, 4, 0, 7, 3], [7, 3, 4, 0, 6, 2], [1, 0, 5, 1, 1, 3]],
    [
        [1, 5, 3, 1, 2, 0],
        [2, 0, 4, 5, 1, 5],
        [2, 6, 4, 5, 2, 0],
        [2, 6, 7, 5, 4, 5],
        [2, 6, 3, 7, 7, 5],
    ],
    [[2, 3, 0, 4, 2, 0], [2, 3, 4, 5, 0, 4], [2, 3, 3, 7, 4, 5], [3, 7, 7, 5, 4, 5]],
    [[3, 2, 7, 3, 7, 5], [7, 5, 5, 4, 3, 2], [3, 2, 5, 4, 1, 0]],
    [
        [2, 3, 0, 4, 2, 0],
        [2, 3, 4, 5, 0, 4],
        [2, 3, 3, 7, 4, 5],
        [3, 7, 7, 5, 4, 5],
        [1, 5, 3, 1, 0, 1],
    ],
    [[3, 2, 1, 5, 3, 1], [3, 2, 5, 4, 1, 5], [3, 2, 7, 5, 5, 4], [3, 7, 7, 5, 3, 2]],
    [[2, 6, 2, 3, 0, 4], [2, 3, 7, 5, 0, 4], [2, 3, 3, 1, 7, 5], [0, 4, 7, 5, 4, 5]],
    [
        [3, 2, 1, 3, 5, 7],
        [5, 7, 6, 2, 3, 2],
        [5, 4, 6, 2, 5, 7],
        [5, 4, 0, 2, 6, 2],
        [5, 4, 1, 0, 0, 2],
    ],
    [
        [4, 5, 0, 4, 2, 6],
        [2, 6, 7, 5, 4, 5],
        [2, 3, 7, 5, 2, 6],
        [2, 3, 1, 5, 7, 5],
        [2, 3, 0, 1, 1, 5],
    ],
    [[2, 3, 2, 0, 2, 6], [1, 5, 7, 5, 4, 5]],
    [[5, 7, 4, 5, 4, 0], [4, 0, 0, 2, 5, 7], [5, 7, 0, 2, 1, 3]],
    [[5, 4, 1, 0, 1, 3], [5, 7, 5, 4, 1, 3]],
    [[0, 2, 4, 5, 0, 4], [0, 2, 5, 7, 4, 5], [0, 2, 1, 5, 5, 7], [0, 1, 1, 5, 0, 2]],
    [[5, 4, 5, 1, 5, 7]],
    [[4, 6, 6, 2, 4, 5], [4, 5, 6, 2, 5, 1], [5, 1, 6, 2, 7, 3]],
    [[4, 6, 6, 2, 4, 5], [4, 5, 6, 2, 5, 1], [5, 1, 6, 2, 7, 3], [0, 2, 4, 0, 0, 1]],
    [[3, 7, 3, 1, 2, 6], [3, 1, 5, 4, 2, 6], [3, 1, 1, 0, 5, 4], [2, 6, 5, 4, 6, 4]],
    [
        [6, 4, 2, 6, 3, 7],
        [3, 7, 5, 4, 6, 4],
        [3, 1, 5, 4, 3, 7],
        [3, 1, 0, 4, 5, 4],
        [3, 1, 2, 0, 0, 4],
    ],
    [[2, 0, 2, 3, 6, 4], [2, 3, 1, 5, 6, 4], [6, 4, 1, 5, 4, 5], [2, 3, 3, 7, 1, 5]],
    [
        [0, 4, 1, 0, 3, 2],
        [3, 2, 6, 4, 0, 4],
        [3, 7, 6, 4, 3, 2],
        [3, 7, 5, 4, 6, 4],
        [3, 7, 1, 5, 5, 4],
    ],
    [
        [1, 3, 0, 1, 4, 5],
        [4, 5, 7, 3, 1, 3],
        [4, 6, 7, 3, 4, 5],
        [4, 6, 2, 3, 7, 3],
        [4, 6, 0, 2, 2, 3],
    ],
    [[3, 7, 3, 1, 3, 2], [5, 4, 6, 4, 0, 4]],
    [[3, 1, 2, 6, 3, 2], [3, 1, 6, 4, 2, 6], [3, 1, 1, 5, 6, 4], [1, 5, 5, 4, 6, 4]],
    [
        [3, 1, 2, 6, 3, 2],
        [3, 1, 6, 4, 2, 6],
        [3, 1, 1, 5, 6, 4],
        [1, 5, 5, 4, 6, 4],
        [0, 4, 1, 0, 2, 0],
    ],
    [[4, 5, 6, 4, 6, 2], [6, 2, 2, 3, 4, 5], [4, 5, 2, 3, 0, 1]],
    [[2, 3, 6, 4, 2, 6], [2, 3, 4, 5, 6, 4], [2, 3, 0, 4, 4, 5], [2, 0, 0, 4, 2, 3]],
    [[1, 3, 5, 1, 5, 4], [5, 4, 4, 6, 1, 3], [1, 3, 4, 6, 0, 2]],
    [[1, 3, 0, 4, 1, 0], [1, 3, 4, 6, 0, 4], [1, 3, 5, 4, 4, 6], [1, 5, 5, 4, 1, 3]],
    [[4, 6, 0, 2, 0, 1], [4, 5, 4, 6, 0, 1]],
    [[4, 6, 4, 0, 4, 5]],
    [[4, 0, 6, 2, 7, 3], [4, 0, 7, 3, 5, 1]],
    [[1, 5, 0, 1, 0, 2], [0, 2, 2, 6, 1, 5], [1, 5, 2, 6, 3, 7]],
    [[3, 7, 1, 3, 1, 0], [1, 0, 0, 4, 3, 7], [3, 7, 0, 4, 2, 6]],
    [[3, 1, 2, 0, 2, 6], [3, 7, 3, 1, 2, 6]],
    [[0, 4, 2, 0, 2, 3], [2, 3, 3, 7, 0, 4], [0, 4, 3, 7, 1, 5]],
    [[3, 7, 1, 5, 1, 0], [3, 2, 3, 7, 1, 0]],
    [[0, 4, 1, 3, 0, 1], [0, 4, 3, 7, 1, 3], [0, 4, 2, 3, 3, 7], [0, 2, 2, 3, 0, 4]],
    [[3, 7, 3, 1, 3, 2]],
    [[2, 6, 3, 2, 3, 1], [3, 1, 1, 5, 2, 6], [2, 6, 1, 5, 0, 4]],
    [[1, 5, 3, 2, 1, 3], [1, 5, 2, 6, 3, 2], [1, 5, 0, 2, 2, 6], [1, 0, 0, 2, 1, 5]],
    [[2, 3, 0, 1, 0, 4], [2, 6, 2, 3, 0, 4]],
    [[2, 3, 2, 0, 2, 6]],
    [[1, 5, 0, 4, 0, 2], [1, 3, 1, 5, 0, 2]],
    [[1, 5, 1, 0, 1, 3]],
    [[0, 2, 0, 1, 0, 4]],
    [],
]
