# Cox: An experimental design and analysis framework

## Introduction
Cox is a lightweight, serverless framework for designing and managing
experiments. Inspired by our own struggles with ad-hoc filesystem-based
experiment collection, and our inability to use heavy-duty frameworks, Cox aims
to be a minimal burden while inducing more organization. Created by [Logan
Engstrom](https://twitter.com/logan_engstrom) and [Andrew
Ilyas](https://twitter.com/andrew_ilyas). 

Cox works by helping you easily __log results__, __collect results__, and
__generate experiments__. For API documentation, see [todo](here); below, we
provide a walkthrough that illustrates the most important features of Cox.

__Why "Cox"? (Aside)__: The name Cox draws both from
[Coxswain](https://en.wikipedia.org/wiki/Coxswain), the person in charge of
steering the boat in a rowing crew, and from the name of [Gertrude
Cox](https://en.wikipedia.org/wiki/Gertrude_Mary_Cox), a pioneer of experimental
design.

## Quick Logging Overview 
The cox logging system is designed for dealing with repeated experiments. The
user defines schemas for [Pandas
dataframes](https://pandas.pydata.org/pandas-docs/stable/reference/api/pandas.DataFrame.html)
that contain all the data necessary for each experiment instance. Each
experiment ran corresponds to a __data store__, and each specified dataframe
from above corresponds to a table within this store. The experiment stores are
organized within the same directory. Cox has a number of utilities for running
and collecting data from experiments of this nature.

## Interactive Introduction

We use Cox most in our machine learning work, but Cox is agnostic to the type or
style of code that you write. To illustrate this, we go through an extremely
simple example in a walkthrough.

## Walkthrough #1: Logging in Cox
__Note__: you can follow along with this tutorial in the [example file here](https://github.com/andrewilyas/cox/blob/master/examples/example.py)!

In this walkthrough, we'll be starting with the following simple piece of code,
which tries to finds the minimum of a quadratic function:

```python
import sys

def f(x):
    return (x - 2.03)**2 + 3

x = ...
tol = ...
step = ...

for _ in range(1000):
    # Take a uniform step in the direction of decrease
    if f(x + step) < f(x - step):
        x += step
    else:
        x -= step

    # If the difference between the directions
    # is less than the tolerance, stop
    if f(x + step) - f(x - step) < tol:
        break
```
### Initializing stores
Logging in Cox is done through the `Store` class, which can be created as follows:
```python
from cox.store import Store
# rest of program here...
store = Store(OUT_DIR)
```

Upon construction, the `Store` instance creates a directory with a random `uuid`
generated name in ```OUT_DIR```, a `HDFStore` for storing data, some logging
files, and a tensorboard directory (named `tensorboard`). Therefore, after we run this command, our `OUT_DIR` directory should look something like this:

```bash
$ ls OUT_DIR
7753a944-568d-4cc2-9bb2-9019cc0b3f49
$ ls 7753a944-568d-4cc2-9bb2-9019cc0b3f49
save        store.h5    tensorboard
```

The experiment ID string `7753a944-568d-4cc2-9bb2-9019cc0b3f49` was
autogenerated. If we wanted to name the experiment something else, we could pass
it as the second parameter; i.e. making a store with `Store(OUT_DIR, 'exp1')`
would make the corresponding experiment ID `exp1`.


### Creating tables
The next step is to declare the data we want to store via _tables_. We can add
arbitrary tables according to our needs, but we need to specify the structure
ahead of time by passing the schema. In our case, we will start out with just a
simple metadata table containing the parameters used to run an instance of the
program above, along with a table for writing the result:

```python
store.add_table('metadata', {
  'step_size': float,
  'tolerance': float, 
  'initial_x': float,
  'out_dir': str
})

store.add_table('result', {
    'final_x': float,
    'final_opt':float
})

```

Each table corresponds exactly to a [Pandas dataframe](https://pandas.pydata.org/pandas-docs/version/0.23.4/generated/pandas.DataFrame.html) found in an `HDFStore`
object.

#### Note on serialization
Cox supports basic object types (like `float`, `int`, `str`, etc) along with any
kind of serializable object (via `dill` or using PyTorch's serialization
method). In particular, if we want to serialize an object we can pass one of the
following types: `cox.store.[OBJECT|PICKLE|PYTORCH_STATE]` as the type value
that is mapped to in the schema dictionary. `cox.store.PYTORCH_STATE` is
particularly useful for dealing with PyTorch objects like weights.
In detail: `OBJECT` corresponds to storing the object as a
serialized string in the table, `PICKLE` corresponds to storing the object as a
serialized string on disk in a separate file, and `PYTORCH_STATE` corresponds to
storing the object as a serialized string on disk using `torch.save`. 

### Logging
Now that we have a table, we can write rows to it! Logging in Cox is done in a
row-by-row manner: at any time, there is a _working row_ that can be appended
to/updated; the row can then be flushed (i.e. written to the file), which starts
a new (empty) working row. The relevant commands are:

```python
# This updates the working row, but does not write it permenantly yet!
store['result'].update_row({
  "final_x": 3.0
})

# This updates it again
store['result'].update_row({
  "final_opt": 3.9409
})

# Write the row permenantly, and start a new working row!
store['result'].flush_row()

# A shortcut for appending a row directly
store['metadata'].append_row({
  'step_size': 0.01,
  'tolerance': 1e-6, 
  'initial_x': 1.0,
  'out_dir': '/tmp/'
}) 
```

#### Incremental updates with `update_row`
Subsequent calls to update_row will edit the same working row. 
This is useful if different parts of the row are computed in different 
functions/locations in the code, as it removes the need for passing statistics 
around all over the place.

### Reading data
By populating tables rows, we are really just adding rows to an underlying
`HDFStore` table. If we want to read the store later, we can simply open another
store at the same location, and then read dataframes with simple commands:

```python
# Note that EXP_ID is the directory the store wrote to in OUT_DIR
s = Store(OUT_DIR, EXP_ID)

# Read tables we wrote earlier
metadata = s['metadata'].df
result = s['result'].df

print(result)
```

Inspecting the `result` table, we see the expected result in our Pandas dataframe!
```
     final_x   final_opt
0   3.000000   3.940900
```

### `CollectionReader`: Reading many experiments at once
Now, in our quadratic example, we aren't just going to try one set of
parameters, we are going to try a number of different values for `step_size`,
`tolerance`, and `initial_x`, as we have not yet discovered convex optimization.
Imagine that we have done so (we will introduce tools for grid searching in the
next section!), and that we have a directory full of stores: (this is why we use
`uuid`s instead of handpicked names!)

```bash
$ ls $OUT_DIR
drwxr-xr-x  6 engstrom  0424807a-c9c0-4974-b881-f927fc5ae7c3
...
...
drwxr-xr-x  6 engstrom  e3646fcf-569b-46fc-aba5-1e9734fedbcf
drwxr-xr-x  6 engstrom  f23d6da4-e3f9-48af-aa49-82f5c017e14f
```

Now, we want to collect all the results from this directory. We can use
`cox.readers.CollectionReader` to read all the tables together in a concatenated
PD table.

```python
from cox.readers import CollectionReader
reader = CollectionReader(OUT_DIR)
print(reader.df('result'))
```

Which gives us all the `result` tables concatenated together as a Pandas
dataframe for easy manipulation:

```
     final_x   final_opt                                exp_id
0   1.000000    4.060900  ed892c4f-069f-4a6d-9775-be8fdfce4713
0   0.000010    7.120859  44ea3334-d2b4-47fe-830c-2d13dc0e7aaa
...
...
0   2.000000    3.000900  f031fc42-8788-4876-8c96-2c1237ceb63d
0 -14.000000  259.960900  73181d27-2928-48ec-9ac6-744837616c4b
```

## Walkthrough Example #2: A Grid Search
One of the most time-saving uses we've found for Cox has been in running grid
searches for optimal hyperparameters. We find ourselves having to do this with
almost every project (e.g. finding the best baseline to compare against).

The way we think about grid searches in Cox is the following: we have a __base
config file__, which contains "default" parameters for our program. On top of
that, we specify a __moving config file__, which describes which parameters we
want to grid over, and what values we should grid over specifically. By default,
Cox will try the Cartesian product of the moving config file, but allows you to
filter out configurations that don't make sense (see below for how).

We start by running the following command:
```
python -m cox.generate-config --path DIRECTORY_WHERE_CONFIGS_ARE_STORED/NAME.py
```

Cox will generate a file called "NAME.py" which is identical to example_config.py in this repository. This is a python file that defines four essential variables:

- ```PARAMS```: A dictionary representing the "moving config". Format is {key: \[list of values to try\]}
- ```NUM_SESSIONS```: An integer, representing the number of sessions to run in parallel
- ```CMD_MAKER```: A function which takes in a tuple (i, d) and returns at string representing the command to run. The first element of the tuple is the index of the job, and ```d``` is a dictionary representing the parameters of the job.
- ```RULES```: A list of boolean functions that take in a dictionary ```d``` representing parameters, and then returns True/False based on whether or not the configuration is valid.

For our example, we might make a config file like this:
```python
from cox.generator import PATH_KEY

gpus = list(range(8))

PARAMS = {
  "tolerance": [0.01, 0.1, 1.0],
  "step_size": [0.1, 0.2, 0.4]
}
NUM_SESSIONS=4
RULES=[lambda d: (d["step_size"] != d["tolerance"])]

CMD_MAKER=lambda i, d: f"CUDA_VISIBLE_DEVICES={gpus[i % len(gpus)]} python main.py --config-path {d[PATH_KEY]}" 
```

Now, for this to work, we need to add some code to our project that parses a json config file. To do this, we can use the `cox.utils.override_json` function, which takes in arguments (in the form of argparse arguments/any class implementing `__getattr__` for all args) and a json path, and returns a new `args` object which has all the parameters from the json file, overridden with any specified args.

```python
from cox.utils import override_json
from argparse import ArgumentParser

parser = ArgumentParser()
parser.add_argument('--config-path', help='path to json config file')
args = parser.parse_args()

args = override_json(args, args.config_path)
```



