import numpy as np

from aiau.strategy.abstract_strategy import AbstractStrategy
from aiau.strategy.selection_utils import select_indices_robust

class BALDStrategy(AbstractStrategy):
    """
    Implementation of the BALD strategy for active learning.
    """

    def __init__(self) -> None:
        super(BALDStrategy, self).__init__()
        self.name = "BALD"

    def select_next_indices(self, data_manager, model, num_suggestions, requery=True, batch_strategy="top-k"):
        """
        Selects the next indices to label.

        Args:
            data_manager (DataManager): Reference to the data_manager which will load the observation if necessary.
            model (Model): Reference to the model which will be used to predict the target values.
            num_suggestions (int): The number of indices to suggest for querying.
            requery (bool): Whether to allow requerying of the same observation.

        Returns:
            list: The indices of the observations to label.
        """
        ensemble_predictions = model.predict(data_manager.full_X)

        # Compute predictive variance across ensemble (epistemic uncertainty)
        var_pred = np.var(ensemble_predictions, axis=0)
        # Compute predictive entropy assuming Gaussian: H = 0.5 * log(2 * pi * e * var)
        # Add small epsilon for numerical stability
        epsilon = 1e-12
        scores = 0.5 * np.log(2 * np.pi * np.e * var_pred + epsilon)

        return select_indices_robust(data_manager, scores, num_suggestions, requery, batch_strategy)