import numpy as np

from aiau.strategy.abstract_strategy import AbstractStrategy
from aiau.strategy.bias_estimation import estimate_bias_squared
from aiau.strategy.selection_utils import select_indices_robust
from aiau.oracles.oracle import Oracle

class BiasReductionStrategy(AbstractStrategy):
    """
    Implementation of the bias reduction strategy for active learning.

    """

    def __init__(self, estimation_approach: str, oracle_type: str, oracle: Oracle) -> None:
        super(BiasReductionStrategy, self).__init__()
        self.name = "BiasReduction"
        self.estimation_approach = estimation_approach
        self.oracle_type = oracle_type
        self.oracle = oracle

    def select_next_indices(self, data_manager, model, num_suggestions, requery=True, batch_strategy="top-k"):
        """
        Selects the next indices to label.

        Args:
            data_manager (DataManager): Reference to the data_manager which will load the observation if necessary.
            model (Model): Reference to the model which will be used to predict the target values.
            oracle (Oracle): Reference to the oracle which will be used to query the target values.
            num_suggestions (int): The number of indices to suggest for querying.
            requery (bool): Whether to allow requerying of the same observation.

        Returns:
            list: The indices of the observations to label.
        """
        output = estimate_bias_squared(data_manager, model, self.estimation_approach, self.oracle_type, self.oracle)
        
        return select_indices_robust(data_manager, output, num_suggestions, requery, batch_strategy)