import numpy as np

from aiau.strategy.abstract_strategy import AbstractStrategy
from aiau.strategy.selection_utils import select_indices_robust

class DiffLeastConfidenceStrategy(AbstractStrategy):
    """
    Implementation of the DiffLeastConfidence strategy for active learning.
    
    """

    def __init__(self) -> None:
        super(DiffLeastConfidenceStrategy, self).__init__()
        self.previous_ensemble_variance = None
        self.name = "DiffLeastConfidence"

    def select_next_indices(self, data_manager, model, num_suggestions, requery=True, batch_strategy="top-k"):
        """
        Selects the next indices to label.

        Args:
            data_manager (DataManager): Reference to the data_manager which will load the observation if necessary.
            model (Model): Reference to the model which will be used to predict the target values.
            num_suggestions (int): The number of indices to suggest for querying.
            requery (bool): Whether to allow requerying of the same observation.

        Returns:
            list: The indices of the observations to label.
        """
        # Predict the target values for all the observations
        ensemble_predictions = model.predict(data_manager.full_X)
        ensemble_variance = np.var(ensemble_predictions, axis=0)

        # For the first round of AL we have no previous model variance or emse so we set them to 0        
        if self.previous_ensemble_variance is None:
            self.previous_ensemble_variance = np.zeros(ensemble_variance.shape)
            output = np.abs(self.previous_ensemble_variance - ensemble_variance)
            self.previous_ensemble_variance = ensemble_variance
        else:
            output = self.previous_ensemble_variance - ensemble_variance
            self.previous_ensemble_variance = ensemble_variance

        return select_indices_robust(data_manager, output, num_suggestions, requery, batch_strategy)