# Implements the committee aka least confidence strategy for active learning.

import numpy as np
from aiau.strategy.abstract_strategy import AbstractStrategy
from aiau.strategy.selection_utils import select_indices_robust

class LeastConfidenceStrategy(AbstractStrategy):
    """
    A committee acquisition strategy that selects the next indices to label based on the least confidence of the ensemble.

    Attributes:
        None

    Methods:
        __init__(self): Initializes a CommitteeLeastConfidenceStrategy instance.
        select_next_indices(self, data_manager, model): Selects the next indices to label.
    """

    def __init__(self) -> None:
        super(LeastConfidenceStrategy, self).__init__()
        self.name = "LeastConfidence"

    def select_next_indices(self, data_manager, model, num_suggestions, requery=True, batch_strategy="top-k"):
        """
        Selects the next indices to label.

        Args:
            data_manager (DataManager): Reference to the data_manager which will load the observation if necessary.
            model (Model): Reference to the model which will be used to predict the target values.
            num_suggestions (int): The number of indices to suggest for querying.
            requery (bool): Whether to allow requerying of the same observation.

        Returns:
            list: The indices of the observations to label.
        """
        # Predict the target values for all the observations
        ensemble_predictions = model.predict(data_manager.full_X)
        
        # Calculate confidence scores (standard deviation across ensemble)
        confidence_scores = np.std(ensemble_predictions, axis=0)

        return select_indices_robust(data_manager, confidence_scores, num_suggestions, requery, batch_strategy)