import numpy as np

from aiau.strategy.abstract_strategy import AbstractStrategy
from aiau.oracles.oracle import Oracle
from aiau.strategy.bias_estimation import estimate_bias_squared
from aiau.strategy.selection_utils import select_indices_robust

class MaxTermSelectionStrategy(AbstractStrategy):
    """
    Implementation of the MaxTermSelection strategy for active learning.
    
    """

    def __init__(self, estimation_approach: str, oracle_type: str, oracle: Oracle) -> None:
        super(MaxTermSelectionStrategy, self).__init__()
        self.name = "MaxTermSelection"
        self.estimation_approach = estimation_approach
        self.oracle_type = oracle_type
        self.oracle = oracle

    def select_next_indices(self, data_manager, model, num_suggestions, requery=True, batch_strategy="top-k"):
        """
        Selects the next indices to label.

        Args:
            data_manager (DataManager): Reference to the data_manager which will load the observation if necessary.
            model (Model): Reference to the model which will be used to predict the target values.
            num_suggestions (int): The number of indices to suggest for querying.
            requery (bool): Whether to allow requerying of the same observation.

        Returns:
            list: The indices of the observations to label.
        """
        # Predict the target values for all the observations
        ensemble_predictions = model.predict(data_manager.full_X)
        ensemble_variance = np.var(ensemble_predictions, axis=0)
        bias_squared = estimate_bias_squared(data_manager, model, self.estimation_approach, self.oracle_type, self.oracle)
        output = np.maximum(bias_squared, ensemble_variance)

        return select_indices_robust(data_manager, output, num_suggestions, requery, batch_strategy)