# Implementation of a random acquisition strategy

import numpy as np
from aiau.strategy.abstract_strategy import AbstractStrategy

class RandomStrategy(AbstractStrategy):
    """
    A random acquisition strategy that selects the next indices to label randomly.

    Attributes:
        None

    Methods:
        __init__(self): Initializes a RandomStrategy instance.
        select_next_indices(self, data_manager, model): Selects the next indices to label.
    """

    def __init__(self) -> None:
        super(RandomStrategy, self).__init__()
        self.name = "Random"

    def select_next_indices(self, data_manager, num_suggestions, requery=True, batch_strategy="top-k"):
        """
        Selects the next indices to label.

        Args:
            data_manager (DataManager): Reference to the data_manager which will load the observation if necessary.
            num_suggestions (int): The number of indices to suggest for querying.
            requery (bool): Whether to allow requerying of the same observation.


        Returns:
            list: The indices of the observations to label.
        """

        if requery:
            # If requery is True, allow the same observation to be selected again
            return np.random.choice(data_manager.indices, num_suggestions, replace=True)
        else:
            unlabelled_indices = set(data_manager.indices) - set(data_manager.labelled_indices)
            if len(unlabelled_indices) < num_suggestions:
                return list(unlabelled_indices)
            else:
                # If requery is False, select from unlabelled indices without replacement
                return np.random.choice(list(unlabelled_indices), num_suggestions, replace=False)