import time
import uuid
from datetime import datetime
import json
from typing import Literal, Optional
from pydantic import BaseModel, Field

class Evidence:
    def __init__(self,
                 text: str,
                 source_type: Literal["message", "hypothesis"],
                 source_id: str):
        self.evidence_id = str(uuid.uuid4())
        self.timestamp = time.time()
        self.text = text
        self.source_type = source_type
        self.source_id = source_id

    def __repr__(self):
        timestamp = datetime.fromtimestamp(self.timestamp).isoformat(sep=' ', timespec='seconds')
        return (
            f"Evidence("
            f"id={self.evidence_id!r}, "
            f"timestamp='{timestamp}', "
            f"source_type={self.source_type!r}, "
            f"source_id={self.source_id!r}, "
            f"text={self.text!r}"
            f")"
        )

    def __str__(self):
        return (
            f"[{self.evidence_id[:8]}] "
            f"{self.text}"
        )
    
    def to_dict(self):
        return {
            "evidence_id": self.evidence_id,
            "timestamp": self.timestamp,
            "source_type": self.source_type,
            "source_id": self.source_id,
            "text": self.text,
        }
    
    def to_json(self):
        return json.dumps(self.to_dict())

class Hypothesis:
    def __init__(self,
                 dimension_id: int,
                 description: str,
                 confidence: float):
        self.hypothesis_id = str(uuid.uuid4())
        self.updated_at = time.time()
        self.dimension_id = dimension_id
        self.description = description
        self.confidence = confidence
        
    def update_confidence(self, new_confidence: float):
        self.confidence = new_confidence
        self.updated_at = time.time()
    
    def update_description(self, new_description: str):
        self.description = new_description
        self.updated_at = time.time()

    def __repr__(self):
        updated = datetime.fromtimestamp(self.updated_at).isoformat(sep=' ', timespec='seconds')
        return (
            f"Hypothesis("
            f"id={self.hypothesis_id!r}, "
            f"dim={self.dimension_id}, "
            f"conf={self.confidence}, "
            f"updated='{updated}', "
            f"desc={self.description!r}"
            f")"
        )

    def __str__(self):
        # A slightly friendlier version
        return (
            f"[{self.hypothesis_id[:8]}] "
            f"dim {self.dimension_id} | conf: {self.confidence} | "
            f"{self.description}"
        )

    def to_dict(self):
        return {
            "hypothesis_id": self.hypothesis_id,
            "updated_at": self.updated_at,
            "dimension_id": self.dimension_id,
            "description": self.description,
            "confidence": self.confidence,
        }

    def to_json(self):
        return json.dumps(self.to_dict())

class Message(BaseModel):
    message_id: str = Field(default_factory=lambda: str(uuid.uuid4()))
    timestamp: float = Field(default_factory=time.time)
    sender: str
    text: str