import json
import time
import os
import uuid
import sys
sys.path.append('../')
import argparse
from typing import List, Dict, Any, Optional, Set
from pydantic import BaseModel
from LLMAgent import LLMBaseAgent
from collections import defaultdict
from MessageGenerator import generate_simple_message, select_best_message_with_rsa, generate_messages_ranked_by_committee, generate_cot_message
from tqdm import tqdm


class SimpleConversationTurn(BaseModel):
    """Single turn in a simple LLM vs LLM conversation"""
    turn_number: int
    speaker: str
    message: str
    
    # Timing and performance info
    processing_time_ms: float = 0.0
    message_generation_time_ms: float = 0.0

    # Optional RSA analysis (when Party A uses RSA/committee)
    rsa_analysis: Dict[str, Any] = {}
    cot_thinking: str = ""


class SimpleConversationResult(BaseModel):
    """Complete conversation result for LLM vs LLM baseline"""
    scenario_id: int
    party_a: str
    party_b: str
    relationship: str
    background_context: str
    turns: List[SimpleConversationTurn]
    conversation_summary: Dict[str, Any]


def load_data(file_path: str) -> List[Dict]:
    """Load conversation data from JSON file"""
    with open(file_path, 'r', encoding='utf-8') as f:
        data = json.load(f)
    return data

def load_processed_ids(files: List[str]) -> Set[int]:
    """Read existing JSONL result files and collect scenario_id values to skip."""
    processed: Set[int] = set()
    for path in files:
        if not path:
            continue
        if not os.path.exists(path):
            continue
        try:
            with open(path, 'r', encoding='utf-8') as f:
                for line in f:
                    line = line.strip()
                    if not line:
                        continue
                    try:
                        obj = json.loads(line)
                    except Exception:
                        continue
                    sid = obj.get('scenario_id')
                    if isinstance(sid, int):
                        processed.add(sid)
        except Exception:
            # If any issue reading a file, ignore and proceed with others
            continue
    return processed

def extract_scenario_fields(s: Dict[str, Any]) -> Dict[str, Any]:
    """Normalize fields from new seed schema to the baseline pipeline structure."""
    scenario_block = s.get('scenario', {}) or {}

    def _first_non_blank(*vals: Any) -> str:
        for v in vals:
            if isinstance(v, str) and v.strip() != "":
                return v.strip()
        return ""

    party_a = _first_non_blank(scenario_block.get('party_a'), s.get('party_a'))
    party_b = _first_non_blank(scenario_block.get('party_b'), s.get('party_b'))
    relationship = _first_non_blank(scenario_block.get('relationship'), s.get('relationship'))
    background_context = _first_non_blank(scenario_block.get('background_context'), s.get('background_context'))

    party_a_background = _first_non_blank(s.get('party_a_background'), s.get('party_a_description'), scenario_block.get('party_a_background'), scenario_block.get('party_a_description'))
    party_b_background = _first_non_blank(s.get('party_b_background'), s.get('party_b_description'), scenario_block.get('party_b_background'), scenario_block.get('party_b_description'))
    # Read party_b_desired_info (replacing party_b_behavior)
    party_b_desired_info = s.get('party_b_desired_info', [])

    # Validate required fields: no None or empty string
    def _is_blank(val: Any) -> bool:
        if val is None:
            return True
        if isinstance(val, str) and val.strip() == "":
            return True
        return False

    normalized = {
        'party_a': party_a,
        'party_b': party_b,
        'relationship': relationship,
        'background_context': background_context,
        'party_a_background': party_a_background,
        'party_b_background': party_b_background,
        'party_b_desired_info': party_b_desired_info,
    }

    # No longer treat party_b_behavior as required; party_b_desired_info is optional and doesn't affect validation
    missing = [k for k, v in normalized.items() if _is_blank(v) and k not in ('party_b_desired_info',)]
    if missing:
        raise ValueError(f"Missing required scenario fields ({', '.join(missing)}) for scenario_id={s.get('scenario_id', 'unknown')}")

    return normalized


# Remove party_b_behavior; add party_b_desired_info to all generation functions
def _extract_rsa_analysis_from_mg_result(result: Any) -> Dict[str, Any]:
    analysis: Dict[str, Any] = {}
    try:
        best_text = None
        best_obj = getattr(result, 'best_candidate', None)
        if best_obj is not None:
            best_text = getattr(best_obj, 'text', None)
            if best_text is None and isinstance(best_obj, dict):
                best_text = best_obj.get('text')
        ranked: List[Dict[str, Any]] = []
        for rc in (getattr(result, 'ranked_candidates', None) or []):
            try:
                rank = getattr(rc, 'rank', None) or (rc.get('rank') if isinstance(rc, dict) else None)
                score = getattr(rc, 'ranking_score', None) or (rc.get('ranking_score') if isinstance(rc, dict) else None)
                cand = getattr(rc, 'candidate', None) or (rc.get('candidate') if isinstance(rc, dict) else None)
                text = getattr(cand, 'text', None) if cand is not None else None
                if text is None and isinstance(cand, dict):
                    text = cand.get('text')
                ranked.append({'rank': rank, 'ranking_score': score, 'text': text})
            except Exception:
                continue
        pairs: List[Dict[str, Any]] = []
        analysis_data = getattr(result, 'analysis_data', None) or (result.get('analysis_data') if isinstance(result, dict) else None)
        if analysis_data is not None:
            for p in (getattr(analysis_data, 'candidate_reply_pairs', None) or analysis_data.get('candidate_reply_pairs') or []):
                try:
                    idx = getattr(p, 'candidate_index', None) or (p.get('candidate_index') if isinstance(p, dict) else None)
                    ctext = getattr(p, 'candidate_text', None) or (p.get('candidate_text') if isinstance(p, dict) else None)
                    replies_out: List[Dict[str, Any]] = []
                    for r in (getattr(p, 'replies', None) or p.get('replies') or []):
                        rtext = getattr(r, 'text', None) or (r.get('text') if isinstance(r, dict) else None)
                        rmeta = getattr(r, 'metadata', None) or (r.get('metadata') if isinstance(r, dict) else {})
                        replies_out.append({'text': rtext, 'metadata': rmeta})
                    pairs.append({'candidate_index': idx, 'candidate_text': ctext, 'replies': replies_out})
                except Exception:
                    continue
        analysis = {'best_candidate_text': best_text, 'ranked_candidates': ranked, 'candidate_reply_pairs': pairs}
    except Exception:
        analysis = {}
    return analysis


def generate_llm_reply(
    party_a: str,
    party_b: str,
    current_speaker: str,
    other_speaker: str,
    relationship: str,
    background_context: str,
    party_a_background: str,
    party_b_background: str,
    conversation_history: List[Dict],
    agent: LLMBaseAgent,
    with_rsa: bool = False,
    with_committee: bool = False,
    with_cot: bool = False,
    party_b_desired_info: Optional[List[Dict[str, Any]]] = None,
) -> str:
    """Generate a reply from the current speaker using LLM with MessageGenerator"""
    # Convert conversation_history to the format expected by MessageGenerator
    prev_convo = []
    for msg in conversation_history:
        prev_convo.append({
            'speaker': msg['speaker'],
            'content': msg['content']
        })
    
    if with_cot:
        # Use CoT message generation for both speakers; only final answer goes into history
        payload = generate_cot_message(
            party_a=party_a,
            party_b=party_b,
            relationship=relationship,
            current_speaker=current_speaker,
            other_speaker=other_speaker,
            background_context=background_context,
            prev_convo=prev_convo,
            agent=agent,
            party_a_background=party_a_background,
            party_b_background=party_b_background,
            is_party_b=(current_speaker == 'Party B'),
            party_b_desired_info=party_b_desired_info,
        )
        text = payload.get('text') if isinstance(payload, dict) else str(payload)
        cot_thinking = payload.get('cot_thinking', '') if isinstance(payload, dict) else ''
        return {'text': text, 'rsa_analysis': {}, 'cot_thinking': cot_thinking}
    elif with_rsa and current_speaker == 'Party A':
        # Use RSA-based message generation for Party A only
        # Get the latest message from the other speaker as new_message
        if conversation_history:
            latest_msg = conversation_history[-1]
            new_message = {
                'text': latest_msg['content'],
                'speaker': latest_msg['speaker']
            }
        else:
            # Fallback simple generation, remove party_b_behavior, pass desired_info (only B will use)
            return generate_simple_message(
                party_a=party_a,
                party_b=party_b,
                relationship=relationship,
                current_speaker=current_speaker,
                other_speaker=other_speaker,
                background_context=background_context,
                prev_convo=prev_convo,
                agent=agent,
                hypotheses=None,
                is_party_b=(current_speaker == 'Party B'),
                party_a_background=party_a_background,
                party_b_background=party_b_background,
                party_b_desired_info=party_b_desired_info,
            )
        
        if with_committee:
            # Use committee-based message generation with RSA
            from MentalModelTypes import Message
            new_message_obj = Message(text=new_message['text'], sender=new_message['speaker'])
            # Pass party_b_desired_info
            result = generate_messages_ranked_by_committee(
                party_a=party_a,
                party_b=party_b,
                relationship=relationship,
                background_context=background_context,
                prev_convo=prev_convo,
                new_message=new_message_obj,
                party_a_background=party_a_background,
                party_b_background=party_b_background,
                hypotheses=None,  # Empty hypotheses for simple LLM
                agent=agent,
                # Simple baseline RSA should not use Party B desired info
                party_b_desired_info=None,
            )
            # Handle both object and dict formats for best_candidate.text
            rsa_analysis = _extract_rsa_analysis_from_mg_result(result)
            if hasattr(result.best_candidate, 'text'):
                text = result.best_candidate.text
            elif isinstance(result.best_candidate, dict) and 'text' in result.best_candidate:
                text = result.best_candidate['text']
            else:
                text = str(result.best_candidate)
            return {'text': text, 'rsa_analysis': rsa_analysis, 'cot_thinking': ''}
        else:
            # Pass party_b_desired_info
            result = select_best_message_with_rsa(
                party_a=party_a,
                party_b=party_b,
                relationship=relationship,
                background_context=background_context,
                prev_convo=prev_convo,
                new_message=new_message,
                party_a_background=party_a_background,
                party_b_background=party_b_background,
                hypotheses=None,
                agent=agent,
                # Simple baseline RSA should not use Party B desired info
                party_b_desired_info=None,
            )
            # Handle both object and dict formats for best_candidate.text
            rsa_analysis = _extract_rsa_analysis_from_mg_result(result)
            if hasattr(result.best_candidate, 'text'):
                text = result.best_candidate.text
            elif isinstance(result.best_candidate, dict) and 'text' in result.best_candidate:
                text = result.best_candidate['text']
            else:
                text = str(result.best_candidate)
            return {'text': text, 'rsa_analysis': rsa_analysis, 'cot_thinking': ''}
    else:
        # Use simple message generation for Party B or when RSA is disabled
        # No longer pass party_b_behavior; uniformly pass party_b_desired_info (B will use, A will ignore)
        text = generate_simple_message(
            party_a=party_a,
            party_b=party_b,
            relationship=relationship,
            current_speaker=current_speaker,
            other_speaker=other_speaker,
            background_context=background_context,
            prev_convo=prev_convo,
            agent=agent,
            hypotheses=None,  # Empty hypotheses as requested
            is_party_b=(current_speaker == 'Party B'),
            party_a_background=party_a_background,
            party_b_background=party_b_background,
            party_b_desired_info=party_b_desired_info,
        )
        return {'text': text, 'rsa_analysis': {}, 'cot_thinking': ''}


def run_simple_conversation(
    scenario_data: Dict,
    max_turns: int = 10,
    agent: LLMBaseAgent = None,
    with_rsa: bool = False,
    with_committee: bool = False,
    with_cot: bool = False,
) -> SimpleConversationResult:
    """
    Run a complete conversation between two simple LLMs (baseline)
    """
    # Extract scenario information
    scenario_id = scenario_data['scenario_id']
    fields = extract_scenario_fields(scenario_data)
    party_a = fields['party_a']
    party_b = fields['party_b']
    relationship = fields['relationship']
    background_context = fields['background_context']
    # Extract party_b_desired_info
    party_b_desired_info = fields.get('party_b_desired_info', [])
    
    # Initialize conversation with first exchange from scenario data
    conversation_history = []
    turns = []
    
    # Add initial conversation from scenario data
    if 'conversation' in scenario_data and isinstance(scenario_data['conversation'], list):
        for msg in scenario_data['conversation']:
            spk = msg.get('speaker', '')
            content = msg.get('content', '')
            if isinstance(spk, str) and "party b" in spk.lower():
                conversation_history.append({'speaker': 'Party B', 'content': content})
                break
            else:
                conversation_history.append({'speaker': 'Party A', 'content': content})
    
    # Import the agent if not provided
    if agent is None:
        raise ValueError("Agent is required")
    
    # Run conversation turns
    turn_progress = tqdm(range(max_turns), desc=f"Scenario {scenario_id} turns", leave=False)
    for turn_num in turn_progress:
        # Determine who should speak next
        last_speaker = conversation_history[-1]['speaker'] if conversation_history else 'Party B'
        
        if last_speaker == 'Party B':
            # Party A's turn
            current_speaker = 'Party A'
            other_speaker = 'Party B'
        else:
            # Party B's turn  
            current_speaker = 'Party B'
            other_speaker = 'Party A'
        
        # Generate message
        if with_cot:
            generation_method = "CoT"
        elif with_rsa and current_speaker == 'Party A':
            generation_method = "Committee" if with_committee else "RSA"
        else:
            generation_method = "Simple"
        turn_progress.set_description(f"Scenario {scenario_id} - {current_speaker} ({generation_method})")
        msg_start_time = time.time()
        
        try:
            payload = generate_llm_reply(
                party_a=party_a,
                party_b=party_b,
                current_speaker=current_speaker,
                other_speaker=other_speaker,
                relationship=relationship,
                background_context=background_context,
                party_a_background=fields.get('party_a_background'),
                party_b_background=fields.get('party_b_background'),
                conversation_history=conversation_history,
                agent=agent,
                with_rsa=with_rsa,
                with_committee=with_committee,
                with_cot=with_cot,
                # Pass desired_info (replacing behavior)
                party_b_desired_info=party_b_desired_info,
            )
            message = payload.get('text') if isinstance(payload, dict) else str(payload)
            rsa_analysis_payload = payload.get('rsa_analysis', {}) if isinstance(payload, dict) else {}
            cot_thinking_payload = payload.get('cot_thinking', '') if isinstance(payload, dict) else ''
        except Exception as e:
            print(f"Error generating message for {current_speaker}: {e}")
            message = f"I'd like to continue our conversation about this topic."
            rsa_analysis_payload = {}
        
        msg_time = (time.time() - msg_start_time) * 1000
        
        # Add to conversation
        conversation_history.append({
            'speaker': current_speaker,
            'content': message
        })
        
        # Record turn
        turn = SimpleConversationTurn(
            turn_number=turn_num + 1,
            speaker=current_speaker,
            message=message,
            processing_time_ms=msg_time,
            message_generation_time_ms=msg_time,
            rsa_analysis=rsa_analysis_payload,
            cot_thinking=cot_thinking_payload
        )
        turns.append(turn)
    
    # Create conversation summary
    summary = {
        'total_turns': len(turns),
        'party_a_turns': len([t for t in turns if t.speaker == 'Party A']),
        'party_b_turns': len([t for t in turns if t.speaker == 'Party B']),
        'avg_message_length': sum(len(t.message) for t in turns) / max(1, len(turns)),
        'total_processing_time_ms': sum(t.processing_time_ms for t in turns),
        'avg_processing_time_ms': sum(t.processing_time_ms for t in turns) / max(1, len(turns))
    }
    
    return SimpleConversationResult(
        scenario_id=scenario_id,
        party_a=party_a,
        party_b=party_b,
        relationship=relationship,
        background_context=background_context,
        turns=turns,
        conversation_summary=summary
    )


def save_result_jsonl(result: SimpleConversationResult, output_file: str):
    """Save single conversation result to JSONL file (append mode)"""
    # Support both Pydantic models and plain dicts
    try:
        if hasattr(result, 'model_dump'):
            result_dict = result.model_dump()
        elif isinstance(result, dict):
            result_dict = result
        elif hasattr(result, 'json'):
            # Fallback for other model-like objects
            result_dict = json.loads(result.json())
        else:
            # Last-resort stringify
            result_dict = {"data": str(result)}
    except Exception:
        # Ensure we can always write something sensible
        try:
            result_dict = json.loads(json.dumps(result, default=lambda o: getattr(o, "__dict__", str(o))))
        except Exception:
            result_dict = {"data": str(result)}
    
    with open(output_file, 'a', encoding='utf-8') as f:
        json.dump(result_dict, f, ensure_ascii=False)
        f.write('\n')  # JSONL format: one JSON object per line


def save_simple_results(results: List[SimpleConversationResult], output_file: str):
    """Save simple conversation results to JSON file (for backward compatibility)"""
    normalized = []
    for result in results:
        try:
            if hasattr(result, 'model_dump'):
                normalized.append(result.model_dump())
            elif isinstance(result, dict):
                normalized.append(result)
            elif hasattr(result, 'json'):
                normalized.append(json.loads(result.json()))
            else:
                normalized.append({"data": str(result)})
        except Exception:
            try:
                normalized.append(json.loads(json.dumps(result, default=lambda o: getattr(o, "__dict__", str(o)))))
            except Exception:
                normalized.append({"data": str(result)})
    results_dict = normalized
    
    with open(output_file, 'w', encoding='utf-8') as f:
        json.dump(results_dict, f, indent=2, ensure_ascii=False)


def save_simple_analysis(results: List[SimpleConversationResult], base_filename: str):
    """Save analysis for simple LLM vs LLM conversations"""
    
    # Create analysis structure
    analysis = {
        "summary": {
            "total_scenarios": len(results),
            "successful_scenarios": len([r for r in results if r.turns]),
            "total_turns": sum(len(r.turns) for r in results),
            "total_party_a_turns": sum(r.conversation_summary['party_a_turns'] for r in results),
            "total_party_b_turns": sum(r.conversation_summary['party_b_turns'] for r in results),
            "avg_turns_per_scenario": sum(len(r.turns) for r in results) / max(1, len(results)),
            "avg_message_length": sum(r.conversation_summary['avg_message_length'] for r in results) / max(1, len(results)),
            "total_processing_time_ms": sum(r.conversation_summary['total_processing_time_ms'] for r in results),
            "avg_processing_time_per_turn_ms": sum(r.conversation_summary['avg_processing_time_ms'] for r in results) / max(1, len(results))
        },
        "scenarios": []
    }
    
    # Process each scenario
    for result in results:
        scenario_analysis = {
            "scenario_id": result.scenario_id,
            "party_a": result.party_a,
            "party_b": result.party_b,
            "relationship": result.relationship,
            "background_context": result.background_context,
            "conversation_flow": [],
            "timing_analysis": {
                "total_time_ms": result.conversation_summary['total_processing_time_ms'],
                "avg_time_per_turn_ms": result.conversation_summary['avg_processing_time_ms'],
                "fastest_turn_ms": min([t.processing_time_ms for t in result.turns]) if result.turns else 0,
                "slowest_turn_ms": max([t.processing_time_ms for t in result.turns]) if result.turns else 0
            },
            "conversation_stats": {
                "total_turns": len(result.turns),
                "party_a_turns": result.conversation_summary['party_a_turns'], 
                "party_b_turns": result.conversation_summary['party_b_turns'],
                "avg_message_length": result.conversation_summary['avg_message_length'],
                "shortest_message": min(len(t.message) for t in result.turns) if result.turns else 0,
                "longest_message": max(len(t.message) for t in result.turns) if result.turns else 0
            }
        }
        
        # Process each turn
        for turn in result.turns:
            turn_info = {
                "turn_number": turn.turn_number,
                "speaker": turn.speaker,
                "message_preview": turn.message,
                "message_length": len(turn.message),
                "processing_time_ms": turn.processing_time_ms
            }
            scenario_analysis["conversation_flow"].append(turn_info)
        
        analysis["scenarios"].append(scenario_analysis)
    
    # Save detailed analysis
    analysis_file = base_filename.replace('.json', '_simple_analysis.json')
    with open(analysis_file, 'w', encoding='utf-8') as f:
        json.dump(analysis, f, indent=2, ensure_ascii=False)
    
    # Save summary report
    summary_file = base_filename.replace('.json', '_simple_summary.txt')
    with open(summary_file, 'w', encoding='utf-8') as f:
        f.write("=== SIMPLE LLM vs LLM CONVERSATION ANALYSIS REPORT ===\n\n")
        f.write(f"Total Scenarios Processed: {analysis['summary']['total_scenarios']}\n")
        f.write(f"Successful Scenarios: {analysis['summary']['successful_scenarios']}\n")
        f.write(f"Total Conversation Turns: {analysis['summary']['total_turns']}\n")
        f.write(f"Total Party A Turns: {analysis['summary']['total_party_a_turns']}\n")
        f.write(f"Total Party B Turns: {analysis['summary']['total_party_b_turns']}\n")
        f.write(f"Average Turns per Scenario: {analysis['summary']['avg_turns_per_scenario']:.2f}\n")
        f.write(f"Average Message Length: {analysis['summary']['avg_message_length']:.1f} characters\n")
        f.write(f"Total Processing Time: {analysis['summary']['total_processing_time_ms']:.1f}ms\n")
        f.write(f"Average Processing Time per Turn: {analysis['summary']['avg_processing_time_per_turn_ms']:.1f}ms\n\n")
        
        f.write("=== PER-SCENARIO SUMMARY ===\n")
        for scenario in analysis["scenarios"]:
            f.write(f"\nScenario {scenario['scenario_id']}:\n")
            f.write(f"  Party A: {scenario['party_a']}\n")
            f.write(f"  Party B: {scenario['party_b']}\n")
            f.write(f"  Relationship: {scenario['relationship']}\n")
            f.write(f"  Total Turns: {scenario['conversation_stats']['total_turns']}\n")
            f.write(f"  Party A/B Turns: {scenario['conversation_stats']['party_a_turns']}/{scenario['conversation_stats']['party_b_turns']}\n")
            f.write(f"  Avg Message Length: {scenario['conversation_stats']['avg_message_length']:.1f} chars\n")
            f.write(f"  Message Length Range: {scenario['conversation_stats']['shortest_message']}-{scenario['conversation_stats']['longest_message']} chars\n")
            f.write(f"  Total Processing Time: {scenario['timing_analysis']['total_time_ms']:.1f}ms\n")
            f.write(f"  Avg Time per Turn: {scenario['timing_analysis']['avg_time_per_turn_ms']:.1f}ms\n")
            f.write(f"  Processing Time Range: {scenario['timing_analysis']['fastest_turn_ms']:.1f}-{scenario['timing_analysis']['slowest_turn_ms']:.1f}ms\n")
    
    return analysis_file, summary_file


def main():
    """Main function to run simple LLM vs LLM conversation pipeline (baseline)"""
    # Parse command line arguments
    parser = argparse.ArgumentParser(description='Run simple LLM vs LLM conversation pipeline')
    parser.add_argument('--model', type=str, default='gpt-4o', 
                        help='Model name to use for the agent (default: gpt-4o)')
    parser.add_argument('--results-dir', type=str, required=True,
                        help='Folder name for saving results')
    parser.add_argument('--temperature', type=float, default=1.0,
                        help='Temperature for the agent (default: 1.0)')
    parser.add_argument('--max-turns', type=int, default=20,
                        help='Maximum number of turns per conversation (default: 20)')
    parser.add_argument('--max-scenarios', type=int, default=200,
                        help='Maximum number of scenarios to process (default: 200)')
    parser.add_argument('--data-file', type=str, required=True,
                        help='Path to the data file')
    parser.add_argument('--with-rsa', action='store_true',
                        help='Use RSA-based message generation for Party A only (default: False)')
    parser.add_argument('--with-committee', action='store_true',
                        help='Use committee-based message generation with RSA for Party A only (default: False)')
    parser.add_argument('--with-cot', action='store_true',
                        help='Use Chain-of-Thought (CI-CoT) message generation for both sides (default: False)')
    
    args = parser.parse_args()
    
    # Load data
    print(f"Loading data from {args.data_file}...")
    scenarios = load_data(args.data_file)
    print(f"Loaded {len(scenarios)} scenarios")
    
    # Create agent with specified model
    print(f"Creating agent with model: {args.model}")
    agent_config = {
        'model': args.model,
        'temperature': args.temperature
    }
    agent = LLMBaseAgent(agent_config)
    
    # Prepare output files BEFORE starting the loop
    timestamp = int(time.time())
    os.makedirs(args.results_dir, exist_ok=True)
    rsa_name = str(args.with_rsa).lower()
    committee_name = str(args.with_committee).lower()
    cot_name = str(args.with_cot).lower()
    output_file = f"{args.results_dir}/simple_llm_conversations_with_rsa_{rsa_name}_committee_{committee_name}_cot_{cot_name}.jsonl"
    conversation_with_meta_data_file = f"{args.results_dir}/simple_llm_conversations_with_meta_data_with_rsa_{rsa_name}_committee_{committee_name}_cot_{cot_name}.jsonl"
    # Do NOT clear existing results; allow resuming and skipping processed scenarios
    print(f"Using JSONL output file: {output_file}")
    print(f"Processing {len(scenarios)} scenarios with simple LLM...")
    print("Results will be written in JSONL format (one scenario per line) as they complete.\n")
    
    # Load already processed scenario ids from existing result files (if present)
    processed_ids = load_processed_ids([output_file, conversation_with_meta_data_file])

    # Run conversations for each scenario - each result saved immediately  
    results = []
    scenario_progress = tqdm(scenarios[:args.max_scenarios], desc="Processing scenarios", unit="scenario")
    
    for scenario in scenario_progress:
        # Validate required fields early; skip if missing
        sid_safe = scenario.get('scenario_id', scenario.get('id', 'unknown'))
        try:
            _ = extract_scenario_fields(scenario)
        except Exception as ve:
            print(f"Skipping scenario {sid_safe}: {ve}")
            continue

        scenario_block = scenario.get('scenario', {}) or {}
        scenario_progress.set_description(f"Scenario {sid_safe} - {scenario_block.get('party_a','')} vs {scenario_block.get('party_b','')}")
        
        # Skip scenarios that already exist in results file(s)
        sid = scenario.get('scenario_id')
        if isinstance(sid, int) and sid in processed_ids:
            print(f"Skipping scenario {sid}: already present in {output_file}")
            continue

        try:
            # The downstream run_simple_conversation will extract and pass party_b_desired_info from fields
            result = run_simple_conversation(scenario, max_turns=args.max_turns, agent=agent, with_rsa=args.with_rsa, with_committee=args.with_committee, with_cot=args.with_cot)
            results.append(result)
            generated_turns = [
                        {"turn_number": t.turn_number, "speaker": t.speaker, "content": t.message}
                        for t in result.turns
                    ]
            scenario['final_conversation'] = generated_turns
            # Immediately save this result to JSONL file
            save_result_jsonl(result, output_file)
            save_result_jsonl(scenario, conversation_with_meta_data_file)
            
        except Exception as e:
            print(f"Error in scenario {sid_safe}: {str(e)}")
            continue
    
    # Save detailed analysis (based on all collected results)
    analysis_file, summary_file = save_simple_analysis(results, output_file)
    
    print(f"\n{'='*60}")
    print("SIMPLE LLM BASELINE PIPELINE COMPLETE")
    print(f"Model used: {args.model}")
    print(f"With RSA (Party A only): {args.with_rsa}")
    print(f"With Committee (Party A only): {args.with_committee}")
    print(f"Processed {len(results)}/{len(scenarios)} scenarios successfully")
    print(f"\nJSONL Format Benefits:")
    print(f"  • Each scenario written immediately upon completion")
    print(f"  • Progressive results available during long runs")
    print(f"  • Easy to resume/continue interrupted experiments")
    print(f"  • Compatible with streaming data processing tools")
    print(f"\nOutput Files:")
    print(f"  • Main Results (JSONL): {output_file}")
    print(f"  • Analysis: {analysis_file}")
    print(f"  • Summary Report: {summary_file}")
    print(f"Analysis complete - all files saved successfully!")
    print(f"{'='*60}")


if __name__ == "__main__":
    main()
