from openai import OpenAI
import numpy as np
import os
from openai import AzureOpenAI
import httpx
from dotenv import load_dotenv

# Create HTTP client with SSL verification disabled
custom_http_client = httpx.Client(verify=False)
load_dotenv(override=True)

# Initialize Azure OpenAI client
emb_client = AzureOpenAI(
    api_version="2024-10-21",
    azure_endpoint=os.getenv("AZURE_OPENAI_API_BASE_URL"),
    api_key=os.getenv("AZURE_OPENAI_API_KEY"),
    http_client=custom_http_client
)

def create_embedding(text: str) -> np.ndarray:
    """
    Create an embedding using OpenAI's text-embedding-3-large (dim=3072).
    Note: text-embedding-3-small is 1536 dimensions.
    Returns an np.ndarray[float32].
    """
    response = emb_client.embeddings.create(
        model="text-embedding-3-large",
        input=text
    )
    emb_list = response.data[0].embedding
    emb_arr = np.array(emb_list, dtype='float32')
    return emb_arr

def cosine_similarity(vec1: np.ndarray, vec2: np.ndarray) -> float:
    """
    Compute cosine similarity between two 1-D numpy arrays.
    Returns a float in [-1, 1].
    """
    # Ensure vectors have the same shape
    if vec1.shape != vec2.shape:
        raise ValueError(f"Shape mismatch: {vec1.shape} vs {vec2.shape}")

    dot = np.dot(vec1, vec2)
    norm1 = np.linalg.norm(vec1)
    norm2 = np.linalg.norm(vec2)
    if norm1 == 0 or norm2 == 0:
        # Avoid division by zero; define similarity with zero-vector as 0.0
        return 0.0

    return dot / (norm1 * norm2)