# The Unreasonable Effectiveness of Randomized Representations in Online Continual Graph Learning

This repository contains the code used in the experiments of the paper "The Unreasonable Effectiveness of Randomized Representations in Online Continual Graph Learning", submitted to ICLR 2026.

## Dependencies

This repository contains our code for OCGL with randomized graph node feature extraction, implemented for running on GPU devices. To run the code, the following packages are required:

* python==3.12.2
* numpy==1.26.4
* torch==2.3.1
* networkx==3.2.1
* scikit-learn==1.4.2
* ogb==1.3.6
* dgl==2.2.1
* torch-geometric==2.6.1
* torch-scatter==2.1.2
* torch-sparse==0.6.18
* pyg-lib==0.4.0

## Usage

To run experiments, you can simply specify the configuration from command line arguments, as an example:

```bash
python main.py --dataset CoraFull \
        --method slda \
        --backbone UGCN \
        --gpu 0 \
        --IL_stream classIL \
	--n_nodes_per_batch 10 \
	--n_validation_tasks 7 \
	--anytime_eval False
```

For baselines with tunable hyper-parameters, we perform a grid search over all the combinations, toghether with learning rate, number of passes on each batch and initialization gain. The candidates, if different from the predefined ones, can be passed through the method-specific argument, such as follows:

```bash
python main.py --dataset Elliptic \
        --method lwf \
	--lwf_args " 'lambda_dist': [0.1, 1.0, 10.0]; 'T': [0.2, 2.0, 20.0]; 'save_every': [10, 100] "
	--lr 0.01,0.001,0.0001
        --backbone GRNF \
        --IL_stream timeIL \
	--n_nodes_per_batch 50 \
	--n_validation_tasks 2
```

## Code overview

We briefly describe here the functioning of the main components of our code, in order to make it more easily understood and modifiable.

### Pipeline

The pipeline is the core of the framework and is implemented in `utils/pipeline.py`. It provides a modular structure for handling different continual learning scenarios, handling how the data stream is presented to the learner and evaluation. The main pipeline classes include:

* `BaseOCGLPipeline`: The base class for all pipelines, providing common functionality such as task processing, evaluation, and model saving.
* `ClassIncremental_OCGLPipeline` and `TimeIncremental_OCGLPipeline`: Specialized pipelines for class-incremental and time-incremental learning scenarios.
* `ClassIncremental_jointPipeline` and `TimeIncremental_jointPipeline`: Variants of the above pipelines for joint training upper bound.

The pipeline is selected based on the `IL_stream` argument provided to the `train.py` script.

### Incremental Graph

The dataset handling is implemented in the `IncrementalGraph` class, located in `utils/dataset.py`. This class manages the evolving graph data, maintaining an up-to-date graph snapshot as new nodes arrive.

### Neighborhood Handler

The neighborhood handler is responsible for processing the node's neighborhood information, in accordance with the requirements of the OCGL setting. Specifically, it subsamples the node neighborhood by leveraging `dgl.dataloading.NeighborSampler`, according to the argument `n_nbs_sample`. It is implemented in the `NeighborhoodProcessor` class in `Backbones/neighborhood_handling.py`. Aside from neighborhood sampling, if the backbone requires it, it uses the provided feature extractor to obtain the node embeddings. The appropriate feature extractor is selected based on the `backbone` argument provided to the pipeline.

To implement a new backbone for feature extraction, you can:

1. Create a new class in the `Backbones` directory.
2. Adjust `Backbones/model_factory.py` to include it.
3. If needed, create a custom `extract()` method for the `NeighborhoodProcessor`.

### Learner

All the considered baselines are extensions of a basic  `OCGLearner`, the core model used for continual graph learning, defined in `Baselines/base_learner.py`. The learner combines a classifier (e.g., a linear layer) with a neighborhood processor for handling graph data. It has functionality to handle online feature centering and to make predictions. The core of the `OCGLearner` is the `observe()` method, which defines how it processes new training data. To make it very easily customizable, the `observe()` method uses four helper methods, one for each stage of training on the mini-batch, which are placeholders to be implemented in specific methods:

* `before_training()`, which can support for example AGEM in computing and storing the gradients on data from the buffer.
* `compute_auxiliary_loss()`, used by most methods such as regularizations ones or ER to obtain the auxiliary term of the loss.
* `after_loss()`, called after the loss computation, for example by AGEM to project the gradients.
* `after_passes()`, used by replay methods to update the buffer, or by regularization methods to update importance scores.

To implement a new strategy, you can:

1. Create a new class in the `Baselines` directory inheriting from  `BaseOCGLearner`.
2. Customize the `observe()` method by simply defining the necessary ones of the above methods.
3. Optionally, modify the `NeighborhoodProcessor`  to handle graph-specific operations.

Refer to the existing implementations in the `Baselines` directory for examples.

## Acknowledgement

This codes extend the OCGL framework defined by the authors of the "Online Continual Graph Learning" paper ([Donghi et al. 2025](https://arxiv.org/abs/2508.03283)). We are also grateful to the authors of [CGLB](https://github.com/QueuQ/CGLB), from which some part of our code was adapted.
