
from gurobipy import Model, GRB

# Create a new model
model = Model("Nutrition_Optimization")

# Create variables
carbohydrates = model.addVar(lb=0, vtype=GRB.CONTINUOUS, name="carbohydrates")
fiber = model.addVar(lb=0, vtype=GRB.INTEGER, name="fiber")
vitamin_e = model.addVar(lb=0, vtype=GRB.CONTINUOUS, name="vitamin_e")
vitamin_b6 = model.addVar(lb=0, vtype=GRB.CONTINUOUS, name="vitamin_b6")
iron = model.addVar(lb=0, vtype=GRB.CONTINUOUS, name="iron")

# Set objective function
model.setObjective(7 * carbohydrates + 5 * fiber + 1 * vitamin_e + 4 * vitamin_b6 + 2 * iron, GRB.MAXIMIZE)

# Resource constraints
resources = {
    'r0': {'description': 'energy stability index', 'upper_bound': 58, 'x0': 3.94, 'x1': 1.81, 'x2': 1.33, 'x3': 3.04, 'x4': 1.14},
    'r1': {'description': 'digestive support index', 'upper_bound': 147, 'x0': 0.66, 'x1': 2.49, 'x2': 4.11, 'x3': 1.64, 'x4': 3.15},
    'r2': {'description': 'immune support index', 'upper_bound': 66, 'x0': 2.94, 'x1': 1.75, 'x2': 2.32, 'x3': 2.78, 'x4': 3.33}
}

for r_key, r_data in resources.items():
    model.addConstr(r_data['x0'] * carbohydrates + r_data['x1'] * fiber + r_data['x2'] * vitamin_e + r_data['x3'] * vitamin_b6 + r_data['x4'] * iron <= r_data['upper_bound'], name=r_key)


# Additional constraints (simplified and deduplicated)
model.addConstr(1.81 * fiber + 1.14 * iron >= 6)
model.addConstr(1.81 * fiber + 3.04 * vitamin_b6 + 1.14 * iron >= 9)
model.addConstr(4.11 * vitamin_e + 1.64 * vitamin_b6 >= 28)
model.addConstr(2.49 * fiber + 4.11 * vitamin_e >= 24)
model.addConstr(0.66 * carbohydrates + 2.49 * fiber >= 21)
model.addConstr(1.75 * fiber + 2.78 * vitamin_b6 + 3.33 * iron >= 10)
model.addConstr(2.94 * carbohydrates + 1.75 * fiber + 2.78 * vitamin_b6 >= 10)
model.addConstr(2.94 * carbohydrates + 2.32 * vitamin_e + 3.33 * iron >= 10)
model.addConstr(2.94 * carbohydrates + 2.78 * vitamin_b6 + 3.33 * iron >= 10)
model.addConstr(1.75 * fiber + 2.32 * vitamin_e + 3.33 * iron >= 10)
model.addConstr(1.75 * fiber + 2.32 * vitamin_e + 2.78 * vitamin_b6 >= 10)
model.addConstr(2.94 * carbohydrates + 1.75 * fiber + 3.33 * iron >= 10)
model.addConstr(2.94 * carbohydrates + 2.32 * vitamin_e + 2.78 * vitamin_b6 >= 10)
model.addConstr(1.81 * fiber + 1.33 * vitamin_e <= 11)
model.addConstr(1.33 * vitamin_e + 3.04 * vitamin_b6 <= 55)


# Optimize model
model.optimize()

# Print results
if model.status == GRB.OPTIMAL:
    print('Obj: %g' % model.objVal)
    for v in model.getVars():
        print('%s %g' % (v.varName, v.x))
elif model.status == GRB.INFEASIBLE:
    print("Model is infeasible")
else:
    print(f"Optimization ended with status {model.status}")

