
import gurobipy as gp

# Create a new model
m = gp.Model("nutrition_optimization")

# Create variables
b1 = m.addVar(lb=0, name="vitamin_b1")  # milligrams of vitamin B1
b5 = m.addVar(lb=0, name="vitamin_b5")  # milligrams of vitamin B5
fat = m.addVar(lb=0, name="fat")  # grams of fat
c = m.addVar(lb=0, name="vitamin_c")  # milligrams of vitamin C

# Set objective function
m.setObjective(7.92 * b1 + 4.9 * b5 + 7.67 * fat + 3.68 * c, gp.GRB.MAXIMIZE)

# Add constraints based on resource indices
cognitive_performance = m.addConstr(5 * b1 + 5 * b5 + 5 * fat + 6 * c >= 56)
digestive_support = m.addConstr(8 * b1 + 7 * b5 + 10 * fat + 12 * c <= 79)
cardiovascular_support = m.addConstr(9 * b1 + 14 * b5 + 2 * fat + 6 * c <= 31)
energy_stability = m.addConstr(13 * b1 + 4 * b5 + 8 * fat + 11 * c <= 222)

# Additional constraints (removing redundant ones)
m.addConstr(5 * b1 + 5 * b5 + 5 * fat + 6 * c <= 238) # Total cognitive performance upper bound
m.addConstr(14 * b5 + 6 * c >= 8) # Combined cardiovascular support for B5 and C
m.addConstr(14 * b5 + 2 * fat >= 10) # Combined cardiovascular support for B5 and fat
m.addConstr(2 * fat + 6 * c >= 4) # Combined cardiovascular support for fat and C
m.addConstr(4 * b5 + 8 * fat >= 45) # Combined energy stability for B5 and fat
m.addConstr(13 * b1 + 4 * b5 >= 25) # Combined energy stability for B1 and B5
m.addConstr(13 * b1 + 8 * fat + 11 * c >= 47) # Combined energy stability for B1, fat, and C


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    print('Vitamin B1:', b1.x)
    print('Vitamin B5:', b5.x)
    print('Fat:', fat.x)
    print('Vitamin C:', c.x)
elif m.status == gp.GRB.INFEASIBLE:
    print('The model is infeasible.')
else:
    print('Optimization ended with status %d' % m.status)

