
import gurobipy as gp
from gurobipy import GRB

try:
    # Create a new model
    m = gp.Model("nutrition_optimization")

    # Create variables
    vitamin_d = m.addVar(lb=0, vtype=GRB.CONTINUOUS, name="vitamin_d")  # milligrams of vitamin D
    protein = m.addVar(lb=0, vtype=GRB.INTEGER, name="protein")  # grams of protein

    # Set objective function
    m.setObjective(8.27 * vitamin_d + 9.25 * protein, GRB.MINIMIZE)

    # Add constraints
    m.addConstr(2.3 * vitamin_d + 0.28 * protein >= 22, "muscle_growth_lower")
    m.addConstr(0.95 * vitamin_d + 1.94 * protein >= 15, "energy_stability_lower")
    m.addConstr(2.92 * vitamin_d + 0.58 * protein >= 12, "immune_support_lower")
    m.addConstr(-8 * vitamin_d + 10 * protein >= 0, "vitamin_d_protein_relation")
    m.addConstr(2.3 * vitamin_d + 0.28 * protein <= 55, "muscle_growth_upper")
    m.addConstr(0.95 * vitamin_d + 1.94 * protein <= 60, "energy_stability_upper")
    m.addConstr(2.92 * vitamin_d + 0.58 * protein <= 44, "immune_support_upper")


    # Optimize model
    m.optimize()

    # Print solution
    if m.status == GRB.OPTIMAL:
        print(f"Optimal objective value: {m.objVal}")
        print(f"Milligrams of Vitamin D: {vitamin_d.x}")
        print(f"Grams of Protein: {protein.x}")
    elif m.status == GRB.INFEASIBLE:
        print("The model is infeasible.")
    else:
        print(f"Optimization ended with status: {m.status}")


except gp.GurobiError as e:
    print(f"Error code {e.errno}: {e}")

except AttributeError:
    print('Encountered an attribute error')
