
from gurobipy import *

# Create a model
m = Model("Optimization_Problem")

# Decision variables
protein_bars = m.addVar(vtype=GRB.INTEGER, name="protein_bars")
ham_sandwiches = m.addVar(vtype=GRB.INTEGER, name="ham_sandwiches")

# Objective function
m.setObjective(5.24 * protein_bars + 6.19 * ham_sandwiches, GRB.MAXIMIZE)

# Constraints
# Total umami index must be at least 47 and no more than 72
m.addConstr(26 * protein_bars + 3 * ham_sandwiches >= 47, name="umami_min")
m.addConstr(26 * protein_bars + 3 * ham_sandwiches <= 72, name="umami_max")

# At least 66 mg of iron
m.addConstr(25 * protein_bars + 16 * ham_sandwiches >= 66, name="iron_min")

# Total iron cannot exceed 121 mg
m.addConstr(25 * protein_bars + 16 * ham_sandwiches <= 121, name="iron_max")

# Constraint: 1 * protein_bars - 7 * ham_sandwiches >= 0
m.addConstr(protein_bars - 7 * ham_sandwiches >= 0, name="protein_ham_ratio")

# Non-negativity constraints (implied by the integer variables but for clarity)
m.addConstr(protein_bars >= 0, name="non_neg_protein")
m.addConstr(ham_sandwiches >= 0, name="non_neg_ham")

# Solve the model
m.optimize()

# Print solution
if m.status == GRB.OPTIMAL:
    print("Optimal solution found:")
    print(f"Protein Bars: {protein_bars.x}")
    print(f"Hams Sandwiches: {ham_sandwiches.x}")
    print(f"Objective Function Value: {m.objVal}")
else:
    print("No optimal solution found")
