
import gurobi

def optimize_fiber_allocation():
    # Create a new Gurobi model
    model = gurobi.Model()

    # Define variables
    granola_bars = model.addVar(name="granola_bars", lb=0)
    eggs = model.addVar(name="eggs", lb=0)
    oreos = model.addVar(name="oreos", lb=0)
    apple_pies = model.addVar(name="apple_pies", lb=0)
    lemons = model.addVar(name="lemons", lb=0)
    kiwis = model.addVar(name="kiwis", lb=0)

    # Objective function coefficients
    obj_coeffs = {
        granola_bars: 3.55,
        eggs: 3.57,
        oreos: 2.41,
        apple_pies: 7.83,
        lemons: 5.78,
        kiwis: 8.69,
    }

    # Set objective function
    model.setObjective(
        obj_coeffs[granola_bars] * granola_bars
        + obj_coeffs[eggs] * eggs
        + obj_coeffs[oreos] * oreos
        + obj_coeffs[apple_pies] * apple_pies
        + obj_coeffs[lemons] * lemons
        + obj_coeffs[kiwis] * kiwis,
        gurobi.GRB.MINIMIZE,
    )

    # Fiber content per item
    fiber_content = {
        granola_bars: 11,
        eggs: 13,
        oreos: 27,
        apple_pies: 2,
        lemons: 32,
        kiwis: 24,
    }

    # Constraints
    # At least 26 grams of fiber must come from granola bars and lemons
    model.addConstraint(granola_bars * fiber_content[granola_bars] + lemons * fiber_content[lemons] >= 26)

    # At least 54 grams of fiber must come from eggs and apple pies
    model.addConstraint(eggs * fiber_content[eggs] + apple_pies * fiber_content[apple_pies] >= 54)

    # At least 50 grams of fiber must come from apple pies and kiwis
    model.addConstraint(apple_pies * fiber_content[apple_pies] + kiwis * fiber_content[kiwis] >= 50)

    # At least 47 grams of fiber must come from granola bars and apple pies
    model.addConstraint(granola_bars * fiber_content[granola_bars] + apple_pies * fiber_content[apple_pies] >= 47)

    # At least 47 grams of fiber must come from all items
    model.addConstraint(
        granola_bars * fiber_content[granola_bars]
        + eggs * fiber_content[eggs]
        + oreos * fiber_content[oreos]
        + apple_pies * fiber_content[apple_pies]
        + lemons * fiber_content[lemons]
        + kiwis * fiber_content[kiwis]
        >= 47
    )

    # 4 times the number of eggs, plus -6 times the number of apple pies should be at least zero
    model.addConstraint(4 * eggs - 6 * apple_pies >= 0)

    # four times the number of eggs, plus minus six times the number of kiwis must be at minimum zero
    model.addConstraint(4 * eggs - 6 * kiwis >= 0)

    # -8 times the number of eggs, plus two times the number of oreos, plus eight times the number of lemons must be greater than or equal to zero
    model.addConstraint(-8 * eggs + 2 * oreos + 8 * lemons >= 0)

    # At most 207 grams of fiber can come from oreos, apple pies, and lemons
    model.addConstraint(oreos * fiber_content[oreos] + apple_pies * fiber_content[apple_pies] + lemons * fiber_content[lemons] <= 207)

    # Total fiber constraint
    model.addConstraint(
        granola_bars * fiber_content[granola_bars]
        + eggs * fiber_content[eggs]
        + oreos * fiber_content[oreos]
        + apple_pies * fiber_content[apple_pies]
        + lemons * fiber_content[lemons]
        + kiwis * fiber_content[kiwis]
        <= 327
    )

    # Solve the model
    model.optimize()

    # Print solution
    if model.status == gurobi.GRB.OPTIMAL:
        print("Optimal solution found.")
        print(f"Granola bars: {granola_bars.varValue}")
        print(f"Eggs: {eggs.varValue}")
        print(f"Oreos: {oreos.varValue}")
        print(f"Apple pies: {apple_pies.varValue}")
        print(f"Lemons: {lemons.varValue}")
        print(f"Kiwis: {kiwis.varValue}")
        print(f"Objective: {model.objVal}")
    else:
        print("No optimal solution found.")

optimize_fiber_allocation()
