
import gurobipy as gp

# Create a new model
m = gp.Model("nutrient_optimization")

# Create variables
potassium = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="potassium")
vitamin_b3 = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="vitamin_b3")
zinc = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="zinc")

# Set objective function
m.setObjective(8.92 * potassium + 6.55 * vitamin_b3 + 4.0 * zinc, gp.GRB.MINIMIZE)

# Add constraints based on resource attributes
resource_data = {
    'r0': {'description': 'immune support index', 'upper_bound': 93, 'x0': 15, 'x1': 2, 'x2': 13},
    'r1': {'description': 'digestive support index', 'upper_bound': 146, 'x0': 7, 'x1': 5, 'x2': 10},
    'r2': {'description': 'muscle growth index', 'upper_bound': 146, 'x0': 7, 'x1': 9, 'x2': 16},
    'r3': {'description': 'energy stability index', 'upper_bound': 253, 'x0': 8, 'x1': 10, 'x2': 17},
    'r4': {'description': 'cardiovascular support index', 'upper_bound': 110, 'x0': 1, 'x1': 12, 'x2': 10}
}

for resource, data in resource_data.items():
    m.addConstr(data['x0'] * potassium + data['x1'] * vitamin_b3 + data['x2'] * zinc <= data['upper_bound'], name=f"{resource}_upper_bound")


# Add additional constraints
m.addConstr(15 * potassium + 13 * zinc >= 10, "immune_support_1")
m.addConstr(2 * vitamin_b3 + 13 * zinc >= 27, "immune_support_2")
m.addConstr(15 * potassium + 2 * vitamin_b3 + 13 * zinc >= 19, "immune_support_3")
m.addConstr(15 * potassium + 2 * vitamin_b3 + 13 * zinc >= 19, "immune_support_4") # Duplicate constraint

m.addConstr(5 * vitamin_b3 + 10 * zinc >= 43, "digestive_support_1")
m.addConstr(7 * potassium + 10 * zinc >= 24, "digestive_support_2")
m.addConstr(7 * potassium + 5 * vitamin_b3 >= 17, "digestive_support_3")
m.addConstr(7 * potassium + 5 * vitamin_b3 + 10 * zinc >= 17, "digestive_support_4")

m.addConstr(7 * potassium + 16 * zinc >= 42, "muscle_growth_1")
m.addConstr(7 * potassium + 9 * vitamin_b3 >= 30, "muscle_growth_2")
m.addConstr(9 * vitamin_b3 + 16 * zinc >= 25, "muscle_growth_3")
m.addConstr(7 * potassium + 9 * vitamin_b3 + 16 * zinc >= 36, "muscle_growth_4")
m.addConstr(7 * potassium + 9 * vitamin_b3 + 16 * zinc >= 36, "muscle_growth_5") # Duplicate constraint

m.addConstr(10 * vitamin_b3 + 17 * zinc >= 64, "energy_stability_1")
m.addConstr(8 * potassium + 10 * vitamin_b3 >= 70, "energy_stability_2")
m.addConstr(8 * potassium + 17 * zinc >= 56, "energy_stability_3")
m.addConstr(8 * potassium + 10 * vitamin_b3 + 17 * zinc >= 56, "energy_stability_4")

m.addConstr(potassium + 10 * zinc >= 36, "cardiovascular_support_1")
m.addConstr(potassium + 12 * vitamin_b3 >= 29, "cardiovascular_support_2")
m.addConstr(potassium + 12 * vitamin_b3 + 10 * zinc >= 29, "cardiovascular_support_3")

m.addConstr(4 * potassium - 7 * vitamin_b3 >= 0, "custom_constraint_1")

m.addConstr(15 * potassium + 13 * zinc <= 58, "immune_support_upper_1")
m.addConstr(15 * potassium + 2 * vitamin_b3 <= 88, "immune_support_upper_2")
m.addConstr(8 * potassium + 17 * zinc <= 147, "energy_stability_upper_1")
m.addConstr(8 * potassium + 10 * vitamin_b3 <= 213, "energy_stability_upper_2")


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    print('potassium:', potassium.x)
    print('vitamin_b3:', vitamin_b3.x)
    print('zinc:', zinc.x)
elif m.status == gp.GRB.INFEASIBLE:
    print("The model is infeasible.")
else:
    print(f"Optimization ended with status {m.status}")

