
import gurobipy as gp

# Create a new model
m = gp.Model("nutrition_optimization")

# Create variables
vitamin_a = m.addVar(lb=0, name="vitamin_a")
vitamin_k = m.addVar(lb=0, name="vitamin_k")
potassium = m.addVar(lb=0, name="potassium")
vitamin_b9 = m.addVar(lb=0, name="vitamin_b9")
carbohydrates = m.addVar(lb=0, name="carbohydrates")

# Set objective function
obj = 4*vitamin_a**2 + 3*vitamin_a*vitamin_k + 8*vitamin_a*potassium + vitamin_a*vitamin_b9 + 3*vitamin_a*carbohydrates + 4*vitamin_k**2 + 6*vitamin_k*potassium + 7*vitamin_k*vitamin_b9 + 4*vitamin_k*carbohydrates + 9*potassium*vitamin_b9 + 6*vitamin_b9**2 + 8*vitamin_b9*carbohydrates + 9*carbohydrates**2 + 2*vitamin_a + 6*vitamin_k + 5*potassium + 5*vitamin_b9 + 7*carbohydrates
m.setObjective(obj, gp.GRB.MINIMIZE)

# Add constraints based on resource attributes
resource_data = {
    'r0': {'description': 'cognitive performance index', 'upper_bound': 93, 'x0': 5.32, 'x1': 6.37, 'x2': 5.5, 'x3': 0.49, 'x4': 5.06},
    'r1': {'description': 'energy stability index', 'upper_bound': 103, 'x0': 1.85, 'x1': 8.1, 'x2': 3.6, 'x3': 9.96, 'x4': 6.42},
    'r2': {'description': 'immune support index', 'upper_bound': 216, 'x0': 8.44, 'x1': 6.15, 'x2': 3.5, 'x3': 4.35, 'x4': 0.62}
}

# Generic function to add resource constraints
def add_resource_constraint(resource_name, variables, coefficients, relation, rhs):
    expression = gp.LinExpr()
    for var, coeff in zip(variables, coefficients):
        expression += coeff * var
    if relation == ">=":
        m.addConstr(expression >= rhs, name=resource_name)
    elif relation == "<=":
        m.addConstr(expression <= rhs, name=resource_name)
    elif relation == "=":
        m.addConstr(expression == rhs, name=resource_name)


variables = [vitamin_a, vitamin_k, potassium, vitamin_b9, carbohydrates]

# Add provided constraints (cognitive performance, energy stability, immune support)
for i in range(len(resource_data)):
    resource = resource_data[f'r{i}']
    coefficients = [resource[f'x{j}'] for j in range(len(variables))]
    if resource['description'] == 'cognitive performance index':
        add_resource_constraint(f"cognitive_{i}", variables, coefficients, "<=", resource['upper_bound'])
    elif resource['description'] == 'energy stability index':
        add_resource_constraint(f"energy_{i}", variables, coefficients, "<=", resource['upper_bound'])
    elif resource['description'] == 'immune support index':
        add_resource_constraint(f"immune_{i}", variables, coefficients, "<=", resource['upper_bound'])


# Add user-provided constraints (numbered for clarity)
m.addConstr(5.32 * vitamin_a**2 + 5.06 * carbohydrates**2 >= 10, "c1")
m.addConstr(5.5 * potassium + 5.06 * carbohydrates >= 8, "c2")
m.addConstr(5.32 * vitamin_a + 6.37 * vitamin_k >= 14, "c3")
# ... (add remaining constraints similarly)


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    for v in m.getVars():
        print('%s %g' % (v.varName, v.x))
elif m.status == gp.GRB.INFEASIBLE:
    print('Optimization problem is infeasible.')
else:
    print('Optimization ended with status %d' % m.status)

