
import gurobipy as gp

try:
    # Create a new model
    m = gp.Model("resource_optimization")

    # Create variables
    bandwidth = m.addVar(vtype=gp.GRB.INTEGER, name="Mbps bandwidth allocated to monitoring")
    patches = m.addVar(vtype=gp.GRB.INTEGER, name="patches per day")
    network_admins = m.addVar(vtype=gp.GRB.INTEGER, name="network administrators")
    system_admins = m.addVar(vtype=gp.GRB.INTEGER, name="system administrators")
    intrusion_analysts = m.addVar(vtype=gp.GRB.INTEGER, name="intrusion analysts")
    security_engineers = m.addVar(vtype=gp.GRB.INTEGER, name="security engineers")

    # Set objective function
    m.setObjective(2 * bandwidth + 1 * patches + 6 * network_admins + 9 * system_admins + 8 * intrusion_analysts + 3 * security_engineers, gp.GRB.MINIMIZE)

    # Add constraints based on power consumption
    power_consumption = {
        'bandwidth': 21,
        'patches': 1,
        'network_admins': 9,
        'system_admins': 23,
        'intrusion_analysts': 15,
        'security_engineers': 5
    }

    m.addConstr(power_consumption['bandwidth'] * bandwidth + power_consumption['patches'] * patches + power_consumption['network_admins'] * network_admins + power_consumption['system_admins'] * system_admins + power_consumption['intrusion_analysts'] * intrusion_analysts + power_consumption['security_engineers'] * security_engineers <= 450, "Total Power Consumption")

    m.addConstr(patches + security_engineers >= 30, "Patches + Security Engineers")
    m.addConstr(bandwidth + system_admins >= 63/23, "Bandwidth + System Admins") # Divided by 23 to match units
    m.addConstr(system_admins + security_engineers >= 63/5, "System Admins + Security Engineers") #Divided by 5 to match units
    m.addConstr(patches + network_admins >= 50/9, "Patches + Network Admins") # Divided by 9 to match units
    m.addConstr(bandwidth + patches >= 43, "Bandwidth + Patches")
    m.addConstr(intrusion_analysts + security_engineers >= 52/5, "Intrusion Analysts + Security Engineers") # Divided by 5 to match units
    m.addConstr(network_admins + system_admins >= 46/9, "Network Admins + System Admins") # Divided by 9 to match units
    m.addConstr(bandwidth + intrusion_analysts >= 55/15, "Bandwidth + Intrusion Analysts") # Divided by 15 to match units
    m.addConstr(bandwidth + patches + network_admins + system_admins + intrusion_analysts + security_engineers >= 55, "Total Resources")
    m.addConstr(-3 * network_admins + 3 * security_engineers >= 0, "Network Admins vs Security Engineers")
    m.addConstr(patches + network_admins <= 354/9, "Patches + Network Admins (Max)") # Divided by 9 to match units
    m.addConstr(bandwidth + patches <= 299, "Bandwidth + Patches (Max)")
    m.addConstr(patches + security_engineers <= 410/5, "Patches + Security Engineers (Max)") # Divided by 5 to match units
    m.addConstr(bandwidth + intrusion_analysts <= 379/15, "Bandwidth + Intrusion Analysts (Max)") # Divided by 15 to match units
    m.addConstr(bandwidth + intrusion_analysts + security_engineers <= 265/5, "Bandwidth + Intrusion Analysts + Security Engineers (Max)") # Divided by 5 to match units
    m.addConstr(network_admins + intrusion_analysts + security_engineers <= 143/5, "Network Admins + Intrusion Analysts + Security Engineers (Max)") # Divided by 5 to match units
    m.addConstr(bandwidth + patches + system_admins <= 272/23, "Bandwidth + Patches + System Admins (Max)") # Divided by 23 to match units
    m.addConstr(patches + system_admins + intrusion_analysts <= 352/15, "Patches + System Admins + Intrusion Analysts (Max)") # Divided by 15 to match units
    m.addConstr(bandwidth + system_admins + intrusion_analysts <= 144/15, "Bandwidth + System Admins + Intrusion Analysts (Max)") # Divided by 15 to match units
    m.addConstr(network_admins + system_admins + intrusion_analysts <= 247/9, "Network Admins + System Admins + Intrusion Analysts (Max)") # Divided by 9 to match units


    # Optimize model
    m.optimize()

    if m.status == gp.GRB.OPTIMAL:
        for v in m.getVars():
            print(f'{v.varName}: {v.x}')
        print(f'Obj: {m.objVal}')
    elif m.status == gp.GRB.INFEASIBLE:
        print("The model is infeasible.")
    else:
        print(f"Optimization ended with status {m.status}")


except gp.GurobiError as e:
    print(f'Error code {e.errno}: {e}')

except AttributeError:
    print('Encountered an attribute error')
