
import gurobi

def optimize_cardiovascular_support():
    # Create a new Gurobi model
    model = gurobi.Model()

    # Define variables
    vitamin_B3 = model.addVar(lb=-gurobi.GRB.INFINITY, ub=gurobi.GRB.INFINITY, name="vitamin_B3")  # milligrams
    fiber = model.addVar(lb=-gurobi.GRB.INFINITY, ub=gurobi.GRB.INFINITY, name="fiber")  # grams
    vitamin_K = model.addVar(lb=-gurobi.GRB.INFINITY, ub=gurobi.GRB.INFINITY, name="vitamin_K")  # milligrams
    carbohydrates = model.addVar(lb=-gurobi.GRB.INFINITY, ub=gurobi.GRB.INFINITY, name="carbohydrates")  # grams

    # Objective function
    model.setObjective(4.8 * vitamin_B3**2 + 2.12 * vitamin_B3 * fiber + 6.97 * vitamin_B3 * vitamin_K + 
                       8.25 * fiber**2 + 7.81 * fiber * vitamin_K + 3.96 * vitamin_B3 + 
                       8.16 * fiber + 7.57 * carbohydrates, gurobi.GRB.MINIMIZE)

    # Constraints
    # The cardiovascular support index for each variable
    model.addConstraint(vitamin_B3 >= 1, name="r0_vitamin_B3")
    model.addConstraint(fiber >= 5, name="r0_fiber")
    model.addConstraint(vitamin_K >= 2, name="r0_vitamin_K")
    model.addConstraint(carbohydrates >= 2, name="r0_carbohydrates")

    # The total combined cardiovascular support index from milligrams of vitamin B3 and milligrams of vitamin K has to be at least 17
    model.addConstraint(vitamin_B3 + vitamin_K >= 17, name="r1")

    # The total combined cardiovascular support index from milligrams of vitamin K squared and grams of carbohydrates squared has to be equal to or greater than 17
    model.addConstraint(vitamin_K**2 + carbohydrates**2 >= 17, name="r2")

    # The total combined cardiovascular support index from milligrams of vitamin B3 and grams of fiber has to be 12 at minimum
    model.addConstraint(vitamin_B3 + fiber >= 12, name="r3")

    # The total combined cardiovascular support index from all variables must be greater than or equal to 12
    model.addConstraint(vitamin_B3 + fiber + vitamin_K + carbohydrates >= 12, name="r4")

    # -5 times the number of milligrams of vitamin B3, plus 3 times the number of grams of fiber must be greater than or equal to zero
    model.addConstraint(-5 * vitamin_B3 + 3 * fiber >= 0, name="r5")

    # -1 times the number of milligrams of vitamin B3 squared, plus 2 times the number of milligrams of vitamin K squared should be greater than or equal to zero
    model.addConstraint(-vitamin_B3**2 + 2 * vitamin_K**2 >= 0, name="r6")

    # minus two times the number of grams of fiber, plus 5 times the number of milligrams of vitamin K, plus 2 times the number of grams of carbohydrates should be no less than zero
    model.addConstraint(-2 * fiber + 5 * vitamin_K + 2 * carbohydrates >= 0, name="r7")

    # The total combined cardiovascular support index from milligrams of vitamin B3 and grams of carbohydrates should be 29 at a maximum
    model.addConstraint(vitamin_B3 + carbohydrates <= 29, name="r8")

    # The total combined cardiovascular support index from milligrams of vitamin B3 and grams of fiber should be 23 at a maximum
    model.addConstraint(vitamin_B3 + fiber <= 23, name="r9")

    # Solve the model
    model.optimize()

    # Print the solution
    if model.status == gurobi.GRB.OPTIMAL:
        print("Optimal Solution:")
        print(f"Milligrams of vitamin B3: {vitamin_B3.varValue}")
        print(f"Grams of fiber: {fiber.varValue}")
        print(f"Milligrams of vitamin K: {vitamin_K.varValue}")
        print(f"Grams of carbohydrates: {carbohydrates.varValue}")
        print(f"Objective: {model.objVal}")
    else:
        print("No optimal solution found.")

optimize_cardiovascular_support()
