
import gurobipy as gp

# Create a new model
m = gp.Model("nutrition_optimization")

# Create variables
potassium = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="potassium")
fiber = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="fiber")
vitamin_d = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="vitamin_d")
vitamin_b12 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_b12")

# Set objective function
m.setObjective(3.17 * potassium + 2.34 * fiber + 2.79 * vitamin_d + 1.75 * vitamin_b12, gp.GRB.MINIMIZE)

# Add constraints based on resource indices
resource_data = {
    'r0': {'description': 'immune support index', 'upper_bound': 478, 'x0': 9, 'x1': 1, 'x2': 15, 'x3': 17},
    'r1': {'description': 'muscle growth index', 'upper_bound': 417, 'x0': 4, 'x1': 6, 'x2': 10, 'x3': 1},
    'r2': {'description': 'cardiovascular support index', 'upper_bound': 255, 'x0': 16, 'x1': 7, 'x2': 2, 'x3': 3}
}

# Resource constraints (upper bounds)
for resource, data in resource_data.items():
    m.addConstr(data['x0'] * potassium + data['x1'] * fiber + data['x2'] * vitamin_d + data['x3'] * vitamin_b12 <= data['upper_bound'], name=f"{resource}_upper_bound")


# Additional constraints from the problem description
m.addConstr(9 * potassium + 1 * fiber >= 61, "immune_constraint1")
m.addConstr(1 * fiber + 15 * vitamin_d >= 98, "immune_constraint2")
m.addConstr(15 * vitamin_d + 17 * vitamin_b12 >= 101, "immune_constraint3")
m.addConstr(9 * potassium + 15 * vitamin_d >= 112, "immune_constraint4")
m.addConstr(9 * potassium + 1 * fiber + 15 * vitamin_d >= 81, "immune_constraint5")
m.addConstr(1 * fiber + 15 * vitamin_d + 17 * vitamin_b12 >= 81, "immune_constraint6")
m.addConstr(9 * potassium + 1 * fiber + 17 * vitamin_b12 >= 81, "immune_constraint7")
m.addConstr(9 * potassium + 1 * fiber + 15 * vitamin_d >= 85, "immune_constraint8")
m.addConstr(1 * fiber + 15 * vitamin_d + 17 * vitamin_b12 >= 85, "immune_constraint9")
m.addConstr(9 * potassium + 1 * fiber + 17 * vitamin_b12 >= 85, "immune_constraint10")
m.addConstr(9 * potassium + 1 * fiber + 15 * vitamin_d >= 61, "immune_constraint11") # Redundant
m.addConstr(1 * fiber + 15 * vitamin_d + 17 * vitamin_b12 >= 61, "immune_constraint12") # Redundant
m.addConstr(9 * potassium + 1 * fiber + 17 * vitamin_b12 >= 61, "immune_constraint13") # Redundant
m.addConstr(9 * potassium + 1 * fiber + 15 * vitamin_d + 17 * vitamin_b12 >= 61, "immune_constraint14") # Redundant

m.addConstr(4 * potassium + 6 * fiber + 10 * vitamin_d + 1 * vitamin_b12 >= 61, "muscle_constraint1")
m.addConstr(2 * vitamin_d + 3 * vitamin_b12 >= 44, "cardio_constraint1")
m.addConstr(16 * potassium + 2 * vitamin_d >= 35, "cardio_constraint2")
m.addConstr(7 * fiber + 2 * vitamin_d + 3 * vitamin_b12 >= 40, "cardio_constraint3")
m.addConstr(16 * potassium + 7 * fiber + 2 * vitamin_d + 3 * vitamin_b12 >= 40, "cardio_constraint4")

m.addConstr(2 * potassium - 5 * vitamin_b12 >= 0, "constraint1")
m.addConstr(-5 * potassium + 5 * vitamin_d >= 0, "constraint2")
m.addConstr(-5 * potassium + 5 * fiber >= 0, "constraint3")

m.addConstr(4 * potassium + 6 * fiber <= 244, "muscle_constraint2")
m.addConstr(4 * potassium + 1 * vitamin_b12 <= 383, "muscle_constraint3")
m.addConstr(6 * fiber + 1 * vitamin_b12 <= 373, "muscle_constraint4")
m.addConstr(4 * potassium + 10 * vitamin_d + 1 * vitamin_b12 <= 113, "muscle_constraint5")
m.addConstr(4 * potassium + 6 * fiber + 1 * vitamin_b12 <= 142, "muscle_constraint6")

m.addConstr(7 * fiber + 3 * vitamin_b12 <= 220, "cardio_constraint5")
m.addConstr(16 * potassium + 3 * vitamin_b12 <= 124, "cardio_constraint6")
m.addConstr(16 * potassium + 7 * fiber + 2 * vitamin_d <= 229, "cardio_constraint7")


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    for v in m.getVars():
        print('%s %g' % (v.varName, v.x))
elif m.status == gp.GRB.INFEASIBLE:
    print("The model is infeasible.")
else:
    print(f"Optimization ended with status {m.status}")

