
import gurobipy as gp

# Create a new model
m = gp.Model("vitamin_optimization")

# Create variables
vitamin_A = m.addVar(lb=0, name="vitamin_A")
vitamin_K = m.addVar(lb=0, name="vitamin_K")
vitamin_C = m.addVar(lb=0, name="vitamin_C")

# Set objective function
m.setObjective(1.0 * vitamin_A + 7.79 * vitamin_K + 9.88 * vitamin_C, gp.GRB.MINIMIZE)

# Add constraints

# Resource constraints (upper bounds are already handled by the problem definition)
muscle_growth = 9 * vitamin_A + 1 * vitamin_K + 10 * vitamin_C
cardiovascular_support = 16 * vitamin_A + 15 * vitamin_K + 11 * vitamin_C
digestive_support = 6 * vitamin_A + 15 * vitamin_K + 6 * vitamin_C
cognitive_performance = 14 * vitamin_A + 5 * vitamin_K + 14 * vitamin_C
immune_support = 15 * vitamin_A + 15 * vitamin_K + 8 * vitamin_C


m.addConstr(muscle_growth >= 68) # Combined constraint supersedes individual A+K and A+C constraints
m.addConstr(cardiovascular_support >= 30) # Combined constraint supersedes individual K+C and A+C constraints
m.addConstr(digestive_support >= 46) # Combined constraint supersedes individual K+C and A+C constraints
m.addConstr(cognitive_performance >= 40) # Combined constraint supersedes individual K+C, A+K, and A+C constraints
m.addConstr(immune_support >= 24) # Combined constraint supersedes individual A+K and A+C constraints

m.addConstr(-4 * vitamin_A + 2 * vitamin_K >= 0)
m.addConstr(10 * vitamin_A - 8 * vitamin_C >= 0)


# Upper bound constraints
m.addConstr(9 * vitamin_A + 10 * vitamin_C <= 211)
m.addConstr(1 * vitamin_K + 10 * vitamin_C <= 140)
m.addConstr(9 * vitamin_A + 1 * vitamin_K <= 213)
m.addConstr(16 * vitamin_A + 11 * vitamin_C <= 172)
m.addConstr(6 * vitamin_A + 6 * vitamin_C <= 89)
m.addConstr(6 * vitamin_A + 15 * vitamin_K + 6 * vitamin_C <= 147)
m.addConstr(5 * vitamin_K + 14 * vitamin_C <= 170)
m.addConstr(14 * vitamin_A + 5 * vitamin_K + 14 * vitamin_C <= 172)
m.addConstr(15 * vitamin_A + 15 * vitamin_K <= 49)



# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    print('Vitamin A: %g' % vitamin_A.x)
    print('Vitamin K: %g' % vitamin_K.x)
    print('Vitamin C: %g' % vitamin_C.x)
elif m.status == gp.GRB.INFEASIBLE:
    print('The model is infeasible.')
else:
    print('Optimization ended with status %d' % m.status)
