
import gurobipy as gp

# Create a new model
m = gp.Model("nutrition_optimization")

# Create variables
fiber = m.addVar(vtype=gp.GRB.INTEGER, name="fiber")
potassium = m.addVar(vtype=gp.GRB.INTEGER, name="potassium")
magnesium = m.addVar(vtype=gp.GRB.CONTINUOUS, name="magnesium")

# Set objective function
m.setObjective(5 * fiber + 1 * potassium + 1 * magnesium, gp.GRB.MINIMIZE)

# Add constraints based on resource attributes
resource_data = {
    'r0': {'description': 'kidney support index', 'upper_bound': 242, 'x0': 8.44, 'x1': 8.9, 'x2': 6.36},
    'r1': {'description': 'energy stability index', 'upper_bound': 232, 'x0': 6.43, 'x1': 2.94, 'x2': 3.65},
    'r2': {'description': 'digestive support index', 'upper_bound': 177, 'x0': 0.91, 'x1': 4.62, 'x2': 8.97},
    'r3': {'description': 'muscle growth index', 'upper_bound': 229, 'x0': 0.9, 'x1': 6.06, 'x2': 8.69},
    'r4': {'description': 'cardiovascular support index', 'upper_bound': 102, 'x0': 4.96, 'x1': 7.58, 'x2': 6.79}
}

# Kidney Support Index
m.addConstr(resource_data['r0']['x0'] * fiber + resource_data['r0']['x2'] * magnesium >= 69)
m.addConstr(resource_data['r0']['x1'] * potassium + resource_data['r0']['x2'] * magnesium >= 37)
m.addConstr(resource_data['r0']['x0'] * fiber + resource_data['r0']['x1'] * potassium + resource_data['r0']['x2'] * magnesium >= 37)

# Energy Stability Index
m.addConstr(resource_data['r1']['x1'] * potassium + resource_data['r1']['x2'] * magnesium >= 56)
m.addConstr(resource_data['r1']['x0'] * fiber + resource_data['r1']['x1'] * potassium >= 49)
m.addConstr(resource_data['r1']['x0'] * fiber + resource_data['r1']['x1'] * potassium + resource_data['r1']['x2'] * magnesium >= 49)
m.addConstr(resource_data['r1']['x1'] * potassium + resource_data['r1']['x2'] * magnesium <= 229)  # Upper bound
m.addConstr(resource_data['r1']['x0'] * fiber + resource_data['r1']['x1'] * potassium <= 200)  # Upper bound


# Digestive Support Index
m.addConstr(resource_data['r2']['x0'] * fiber + resource_data['r2']['x1'] * potassium >= 33)
m.addConstr(resource_data['r2']['x0'] * fiber + resource_data['r2']['x1'] * potassium + resource_data['r2']['x2'] * magnesium >= 33)

# Muscle Growth Index
m.addConstr(resource_data['r3']['x0'] * fiber + resource_data['r3']['x1'] * potassium >= 50)
m.addConstr(resource_data['r3']['x0'] * fiber + resource_data['r3']['x2'] * magnesium >= 56)
m.addConstr(resource_data['r3']['x1'] * potassium + resource_data['r3']['x2'] * magnesium >= 62)
m.addConstr(resource_data['r3']['x0'] * fiber + resource_data['r3']['x1'] * potassium + resource_data['r3']['x2'] * magnesium >= 62)

# Cardiovascular Support Index
m.addConstr(resource_data['r4']['x0'] * fiber + resource_data['r4']['x2'] * magnesium >= 17)
m.addConstr(resource_data['r4']['x0'] * fiber + resource_data['r4']['x1'] * potassium + resource_data['r4']['x2'] * magnesium >= 17)
m.addConstr(resource_data['r4']['x1'] * potassium + resource_data['r4']['x2'] * magnesium <= 97) # Upper bound

# Additional Constraints
m.addConstr(-2 * potassium + 9 * magnesium >= 0)
m.addConstr(-6 * fiber + 3 * potassium >= 0)


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    print('fiber:', fiber.x)
    print('potassium:', potassium.x)
    print('magnesium:', magnesium.x)
elif m.status == gp.GRB.INFEASIBLE:
    print('Model is infeasible')
else:
    print('Optimization ended with status %d' % m.status)
